/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
//         Dual Ray Smoke Sensor Software Library
//
// This library implements functions to perform measurements for smoke detector
//  AFE using dual-ray approach.
// Texas Instruments, Inc.
// Luis R
// *****************************************************************************
#include <DualRaySmokeAFELib.h>
#include <DualRaySmokeAFELib_Config.h>
#include <DualRaySmokeAFE_HAL.h>

/**** Local Variables *********************************************************/
// State of Alarm
static volatile tDualRaySmokeAFE_AlarmDetectionState tAlarmState;
// Pointer to HAL Configuration structure
tDualRaySmokeAFE_HAL_Config *pHALConfig;
// Pointer to Library Configuration structure
tDualRaySmokeAFELib_Config *pLibConfig;

/**** Functions ***************************************************************/
void DualRaySmokeAFELib_Init(tDualRaySmokeAFELib_Config *LibConfig,
                             tDualRaySmokeAFE_HAL_Config *HALConfig)
{
    tAlarmState = DUALRAYSMOKEAFE_ALARM_OFF;
    pHALConfig = HALConfig;
    pLibConfig = LibConfig;
    DualRaySmokeAFE_HAL_AFE_Init(pHALConfig->AFEConfig);
    DualRaySmokeAFE_HAL_Sounder_Init(pHALConfig->sounderConfig);
}

void DualRaySmokeAFELib_UpdateConfig(tDualRaySmokeAFELib_Config *LibConfig,
                                     tDualRaySmokeAFE_HAL_Config *HALConfig)
{
    pHALConfig = HALConfig;
    pLibConfig = LibConfig;
    DualRaySmokeAFE_HAL_AFE_Init(pHALConfig->AFEConfig);
    DualRaySmokeAFE_HAL_Sounder_Init(pHALConfig->sounderConfig);
}

tDualRaySmokeAFE_AlarmDetectionState DualRaySmokeAFELib_DetectionAlgorithm(
                                     tDualRaySmokeAFELib_DualRayData *Data)
{
    uint8_t threshold_high = 0;

    if (pLibConfig->bAlarmDetectionEnabled == false)
    {
        // Do not run the alarm detection algorithm if disabled
        return tAlarmState;
    }

#if (DUALRAYSMOKEAFELIB_ENABLE_INFRAREDLED_DETECTION != 0)
    // Check if photodiode detected light with infrared LED
    if ( Data->sLED[DUALRAYSMOKEAFE_LED_RED].i16Diff >
              pLibConfig->i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_RED] )
        {
            threshold_high |= (1 << DUALRAYSMOKEAFE_LED_RED);
        }
#endif
#if (DUALRAYSMOKEAFELIB_ENABLE_BLUELED_DETECTION != 0)
    // Check if photodiode detected light with blue LED
    if ( (Data->sLED[DUALRAYSMOKEAFE_LED_BLUE].i16Diff  >
                  Config->i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_BLUE] )
        {
            threshold_high |= (1 << DUALRAYSMOKEAFE_LED_BLUE);
        }
#endif

    // State machine based on Alarm state
    switch (tAlarmState)
    {
        case DUALRAYSMOKEAFE_ALARM_OFF:
            // Alarm is OFF, if the threshold is high, go to warning 1
            // Otherwise, keep checking
            if (threshold_high != 0)
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_WARNING1;
            }
        break;
        case DUALRAYSMOKEAFE_ALARM_WARNING1:
            // Alarm is in Warning #1, if the threshold is still high,
            // go to warning 2;
            // Otherwise, go back to OFF state
            if (threshold_high != 0)
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_WARNING2;
            }
            else
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_TURN_OFF;
            }
        break;
        case DUALRAYSMOKEAFE_ALARM_WARNING2:
            // Alarm is in Warning #2, if the threshold is still high,
            // turn ON alarm;
            // Otherwise, go back to OFF state
            // Note: the application must call DualRaySmokeAFELib_AlarmOn to
            //      actually move to ON state
            if (threshold_high != 0)
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_TURN_ON;
            }
            else
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_TURN_OFF;
            }
        break;
        case DUALRAYSMOKEAFE_ALARM_ON:
        case DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDHIGH:
        case DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDLOW:
            // Alarm is ON, if the threshold is still high, keep alarm ON
            if (threshold_high != 0)
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDHIGH;
            }
            else
            {
                tAlarmState = DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDLOW;
            }
        break;
        case DUALRAYSMOKEAFE_ALARM_SILENT:
            // Alarm is silent, don't do anything until it's back to OFF
        break;
    }

    return tAlarmState;
}

tDualRaySmokeAFE_AlarmDetectionState DualRaySmokeAFELib_AlarmGetState(void)
{
    return tAlarmState;
}

void DualRaySmokeAFELib_setAlarmSilent(void)
{
    tAlarmState = DUALRAYSMOKEAFE_ALARM_SILENT;
}

void DualRaySmokeAFELib_setAlarmOn(bool sounderOn)
{
    tAlarmState = DUALRAYSMOKEAFE_ALARM_ON;
    if (sounderOn == true)
    {
        DualRaySmokeAFE_HAL_Sounder_SetOn();
    }
}

void DualRaySmokeAFELib_setAlarmOff(void)
{
    tAlarmState = DUALRAYSMOKEAFE_ALARM_OFF;
    DualRaySmokeAFE_HAL_Sounder_SetOff();
}

