/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
//  Dual Ray AFE HAL Configuration
//
//  Static and default configuration of Library
// Texas Instruments, Inc.
// Luis R
#ifndef __DUALRAYSMOKEAFE_HAL_CONFIG_FR235X_SACL3_H__
#define __DUALRAYSMOKEAFE_HAL_CONFIG_FR235X_SACL3_H__

#include <msp430.h>
#include <stdint.h>
#include <stdbool.h>

//*********** AFE ************************************************************
//! \brief Number of samples for each LED measurement
//!         Use power of 2 for simpler/faster division/shift
//!
#define DUALRAYSMOKEAFE_HAL_AFE_LED_SAMPLE_BUFFER_SIZE  (4)

//! \brief Setting of internal VREF in mV for Smoke Measurements
//!         i.e. 2500 = 2.5V
//!
#define DUALRAYSMOKEAFE_HAL_AFE_VREF_MV                 (2500)

//! \brief Delay in us after Internal Reference and SAC is enabled
//!         for 1st measurement.
//!         i.e. 120 = 120us
//!
#define DUALRAYSMOKEAFE_HAL_AFE_DELAY_MEASUREMENT1      (120)

//! \brief Delay in us after SAC is enabled for 2nd measurement.
//!         i.e. 150 = 150us
//!
#define DUALRAYSMOKEAFE_HAL_AFE_DELAY_MEASUREMENT2      (120)

//! \brief Setting of internal VREF in mV for Temperature Measurements
//!         i.e. 1500 = 1.5V
//!
#define DUALRAYSMOKEAFE_HAL_AFE_TEMPERATURE_VREF_MV     (1500)

//! \brief Delay in us for internal reference for temperature measurement.
//!         i.e. 100 = 100us
//!
#define DUALRAYSMOKEAFE_HAL_AFE_DELAY_TEMPERATURE       (100)

//*********** I/O **************************************************************
//! \brief Port bit for Indicator LED
//!
#define  DUALRAYSMOKEAFE_HAL_IO_LED_INDICATOR_BIT       (BIT4)

//! \brief Port Direction register for Indicator LED
//!
#define  DUALRAYSMOKEAFE_HAL_IO_LED_INDICATOR_PDIR      (P3DIR)

//! \brief Port Output register for Indicator LED
//!
#define  DUALRAYSMOKEAFE_HAL_IO_LED_INDICATOR_POUT      (P3OUT)

//! \brief Port bit for Alarm output pin
//!
#define  DUALRAYSMOKEAFE_HAL_IO_ALARM_OUTPUT_BIT        (BIT3)

//! \brief Port Direction register for Alarm output pin
//!
#define  DUALRAYSMOKEAFE_HAL_IO_ALARM_OUTPUT_PDIR       (P2DIR)

//! \brief Port Output register for Alarm output pin
//!
#define  DUALRAYSMOKEAFE_HAL_IO_ALARM_OUTPUT_POUT       (P2OUT)

//! \brief Port bit for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_BIT          (BIT5)

//! \brief Port Direction register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PDIR         (P2DIR)

//! \brief Port Input register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PIN          (P2IN)

//! \brief Port Output register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_POUT         (P2OUT)

//! \brief Interrupt Enable register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PIE          (P2IE)

//! \brief Interrupt Edge register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PIES         (P2IES)

//! \brief Pull Resistor register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PREN         (P2REN)

//! \brief Port flag register for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PIFG         (P2IFG)

//! \brief Port vector for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_VECTOR       (PORT2_VECTOR)

//! \brief Port interrupt vector for Test Button
//!
#define  DUALRAYSMOKEAFE_HAL_IO_TESTBUTTON_PIV          (P2IV)

//! \brief Number of timer events to detect long press
//!         I.e. 20 = 20x100ms = 2sec
#define DUALRAYSMOKEAFE_HAL_IO_TESTBUTTONTIMER_LONGPRESSCYCLES (20)

//*********** Sounder **********************************************************
//! \brief Sounder Period in LSBUS_FREQ_HZ cycles and using Up/Down PWM
//!         PWM Freq = Clock source / (period *2)
//!         i.e. 32768 / (33*2) = ~496Hz
//!
#define  DUALRAYSMOKEAFE_HAL_SOUNDER_PERIOD                 (33)

//! \brief Sounder Duty cycle in LSBUS_FREQ_HZ cycles and using Up/Down PWM
//!         PWM Duty cycle % = Duty/Period
//!         i.e. 16 / (33) = ~50%
//!
#define  DUALRAYSMOKEAFE_HAL_SOUNDER_DUTY                   (16)

//*********** GUI Communication ************************************************
//! \brief baudrate of GUI Communication port
//!         i.e. 19200 = 19,200bps
//!
#define DUALRAYSMOKEAFE_HAL_GUICOMM_BAUDRATE                (19200)

//*********** Timing ULP Timer (RTC) *******************************************
// Periodic interrupt working at lowest power consumption.
//  Uses RTC with VLO source.
//  Can be calibrated for higher accuracy using GP Timer.
//  Expected to be ON all the time.

//! \brief ULP Timer frequency in Hz
//!         RTC is set to VLO/10 for this demo
//!
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_FREQ_HZ         (1000)

//! \brief Source for ULP Timer in HZ
//!         Sourced by VLO
//!
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_SOURCECLK_HZ    (10000)

//! \brief Minimum Interval for ULP Timer in ms
//!
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_MIN_INTERVAL_MS (1)

//! \brief Maximum Interval for ULP Timer in ms
//!
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_MAX_INTERVAL_MS (30000)

//! \brief ULP Timer cycles used to calibrate VLO
//!
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_CAL_CYCLES      (50)

//! \brief Interrupt Vector for ULP Timer
#define DUALRAYSMOKEAFE_HAL_TIMING_ULPTIMER_VECTOR          (RTC_VECTOR)

//*********** Timing GP Timer **************************************************
// General purpose timer used to create us and ms blocking
//  delays, and calibrate ULP Timer.
//  us delay and calibration use ACLK (REFO).
//  ms delay uses SMCLK (DCO).
//  Independent from ULP and LP since it needs to run

//! \brief Macro used to configure GP Timer for Capture function.
//!         Capture at CCI0A on RTC output.
//!         Capture at rising edge, SCS=0 asynchronous capture,
//!         SCS=1 synchronous capture. Enable capture mode interrupt.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_CAPTUREINIT()  \
                                    {TB0CCTL0 = CM_1 | CCIS_0 | CAP | CCIE;}

//! \brief Macro used to start GP Timer in capture mode.
//!         Use ACLK as clock source, clear TB0R, divider=/1,
//!         no Timer TBIFG interrupt.
//!         Start timer in continuous mode - Timer counts up to the value of
//!         CNTL-0xFFFF.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_CAPTURESTART() \
                               {TB0CTL = TBSSEL__ACLK | MC__CONTINUOUS | TBCLR;}

//! \brief Macro used to get value of CCR0 after a capture.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_CAPTUREGETCCR() (TB0CCR0)

//! \brief Macro used to configure GP Timer for Compare function.
//!         Generates event based on x interval
//! \param[in] x is the interval in timer source (ACLK) cycles
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_COMPAREINIT(x)  \
                                            {TB0CCTL0 = CCIE; TB0CCR0 = x;}

//! \brief Macro used to start GP Timer in Low-Power compare mode.
//!         Use ACLK as clock source, clear TB0R, divider=/1,
//!         no Timer TBIFG interrupt.
//!         Start timer in up mode - Timer counts up to the value of CCR0.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_LPCOMPARESTART()  \
                            {TB0CTL = TBSSEL__ACLK | MC__UP | TBCLR;}

//! \brief Macro used to start GP Timer in High Precision compare mode.
//!         Use SMCLK as clock source, clear TB0R, divider=/1,
//!         no Timer TBIFG interrupt.
//!         Start timer in up mode - Timer counts up to the value of CCR0.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_HPCOMPARESTART()  \
                            {TB0CTL = TBSSEL__SMCLK | MC__UP | TBCLR;}

//! \brief Macro used to Stop Timer and clear Timer channel settings.
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_STOP()  {TB0CTL = 0; TB0CCTL0 = 0;}

//! \brief Interrupt Vector for General Purpose Timer
#define DUALRAYSMOKEAFE_HAL_TIMING_GPTIMER_VECTOR   (TIMER0_B0_VECTOR)

//*********** Timing LP Periodic Timer******************************************
// Periodic timer independent from ULP Timer and GP Timer.
//  Used to create multiple periodic events at a predefined
//  time base (i.e. 100ms).
//  Uses ACLK (REFO) which is more accurate but requires
//  more power, so it shouldn't be ON all the time.

//! \brief Macro used to configure LP Periodic Timer to generate periodic
//!         event.
//!         Use ACLK as clock source, clear TBxR,
//!         enable interrupt.
//!         Start timer in up mode - Timer counts up to the value of CCR0.
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_INIT(X)  \
                                {TB1CCTL0 = 0; \
                                TB1CCR0 = X;\
                                TB1CCTL0 = CCIE;\
                                TB1CTL = TBSSEL__ACLK | MC__STOP | TBCLR;}

//! \brief Macro used to restart LP Periodic Timer
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_RESTART()  \
                                {TB1CTL = TBSSEL__ACLK | MC__STOP | TBCLR;\
                                TB1CTL = TBSSEL__ACLK | MC__UP | TBCLR;}

//! \brief Macro used to run or keep running LP Periodic Timer
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_RUN()  \
                                {TB1CTL = TBSSEL__ACLK | MC__UP;}

//! \brief Macro used to stop LP Periodic Timer.
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_STOP()  \
                                {TB1CTL = TBSSEL__ACLK | MC__STOP | TBCLR;\
                                TB1CCTL0 &= ~CCIFG; }

//! \brief Interrupt Vector for LP Periodic Timer.
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_VECTOR    (TIMER1_B0_VECTOR)

//! \brief Maximum number of callback functions supported by LP Periodic Timer.
#define DUALRAYSMOKEAFE_HAL_TIMING_LPPERTIMER_CALLBACK_MAX  (3)

#endif //__DUALRAYSMOKEAFE_HAL_CONFIG_FR235X_SACL3_H__
