/*
 * decode.h
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2005
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

#ifndef _DECODE_H
#define _DECODE_H

/* Standard Linux headers */
#include <stdio.h>
#include <pthread.h>

/* Enables or disables debug output */
#ifdef __DEBUG
#define DBG(fmt, args...) fprintf(stderr, "Decode Debug: " fmt, ## args)
#else
#define DBG(fmt, args...)
#endif

#define ERR(fmt, args...) fprintf(stderr, "Decode Error: " fmt, ## args)

/* Function error codes */
#define SUCCESS         0
#define FAILURE         -1

/* Thread error codes */
#define THREAD_SUCCESS  (void *) 0
#define THREAD_FAILURE  (void *) -1

/* The codec engine to use in this application */
#define ENGINE_NAME     "decode"

/* Device parameters */
#define V4L2_DEVICE     "/dev/video0"
#define OSD_DEVICE      "/dev/fb/0"
#define ATTR_DEVICE     "/dev/fb/2"
#define FBVID_DEVICE    "/dev/fb/3"
#define SOUND_DEVICE    "/dev/dsp"
#define FBVID_HD_DEVICE "/dev/fb/1"

/* True of false enumeration */
#define TRUE            1
#define FALSE           0

/* Scaling factors for the video standards */
#define NOSTD           0
#define PAL             12
#define NTSC            10

/* Masks to determine when clips are done playing */
#define VIDEO_DONE      0x1
#define SOUND_DONE      0x2

/* Number of microseconds to pause if the demo is 'paused' */
#define PAUSE           100

typedef enum SpeechDecoder {
    G711_SPEECH_DECODER,
    NUM_SPEECH_DECODERS
} SpeechDecoder;

typedef enum AudioDecoder {
    MPEG2AAC_AUDIO_DECODER,
    MPEG1L2_AUDIO_DECODER,
    NUM_AUDIO_DECODERS
} AudioDecoder;

typedef enum VideoDecoder {
    MPEG4_VIDEO_DECODER,
    H264_VIDEO_DECODER,
    MPEG2_VIDEO_DECODER,
    NUM_VIDEO_DECODERS
} VideoDecoder;

typedef enum Resolution {
    RES_UNDEFINED,
    RES_NTSC,
    RES_PAL,
    RES_VGA,
    RES_720P,
    RES_1080I
} Resolution;

/* Global data structure */
typedef struct GlobalData {
    int             quit;                /* Global quit flag */
    int             play;                /* Whether to play or pause */
    int             frames;              /* Video frame counter */
    int             videoBytesEncoded;   /* Video bytes encoded counter */
    int             soundBytesEncoded;   /* Sound bytes encoded counter */
    int             samplingFrequency;   /* Sound sampling frequency */
    int             imageWidth;          /* Width of clip */
    int             imageHeight;         /* Height of clip */
    unsigned int    doneMask;            /* Mask of clips done playing */
    unsigned int    targetMask;          /* Mask of required clips done */
    int             done;                /* Flag to indicate clips are done */
    int             yFactor;             /* Vertical scaling (PAL vs. NTSC) */
    pthread_mutex_t mutex;               /* Mutex to protect the global data */
} GlobalData;

/* Global data */
extern GlobalData gbl;

/* Functions to protect the global data */
static inline int gblGetQuit(void)
{
    int quit;

    pthread_mutex_lock(&gbl.mutex);
    quit = gbl.quit;
    pthread_mutex_unlock(&gbl.mutex);

    return quit;
}

static inline void gblSetQuit(void)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.quit = TRUE;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetPlay(void)
{
    int play;

    pthread_mutex_lock(&gbl.mutex);
    play = gbl.play;
    pthread_mutex_unlock(&gbl.mutex);

    return play;
}

static inline void gblSetPlay(int play)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.play = play;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetAndResetFrames(void)
{
    int frames;

    pthread_mutex_lock(&gbl.mutex);
    frames = gbl.frames;
    gbl.frames = 0;
    pthread_mutex_unlock(&gbl.mutex);

    return frames;
}

static inline void gblIncFrames(void)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.frames++;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetAndResetVideoBytesEncoded(void)
{
    int videoBytesEncoded;

    pthread_mutex_lock(&gbl.mutex);
    videoBytesEncoded = gbl.videoBytesEncoded;
    gbl.videoBytesEncoded = 0;
    pthread_mutex_unlock(&gbl.mutex);

    return videoBytesEncoded;
}

static inline void gblIncVideoBytesEncoded(int videoBytesEncoded)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.videoBytesEncoded += videoBytesEncoded;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetAndResetSoundBytesEncoded(void)
{
    int soundBytesEncoded;

    pthread_mutex_lock(&gbl.mutex);
    soundBytesEncoded = gbl.soundBytesEncoded;
    gbl.soundBytesEncoded = 0;
    pthread_mutex_unlock(&gbl.mutex);

    return soundBytesEncoded;
}

static inline void gblIncSoundBytesEncoded(int soundBytesEncoded)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.soundBytesEncoded += soundBytesEncoded;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetSamplingFrequency(void)
{
    int samplingFrequency;

    pthread_mutex_lock(&gbl.mutex);
    samplingFrequency = gbl.samplingFrequency;
    pthread_mutex_unlock(&gbl.mutex);

    return samplingFrequency;
}

static inline void gblSetSamplingFrequency(int samplingFrequency)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.samplingFrequency = samplingFrequency;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetImageWidth(void)
{
    int imageWidth;

    pthread_mutex_lock(&gbl.mutex);
    imageWidth = gbl.imageWidth;
    pthread_mutex_unlock(&gbl.mutex);

    return imageWidth;
}

static inline void gblSetImageWidth(int imageWidth)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.imageWidth = imageWidth;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetImageHeight(void)
{
    int imageHeight;

    pthread_mutex_lock(&gbl.mutex);
    imageHeight = gbl.imageHeight;
    pthread_mutex_unlock(&gbl.mutex);

    return imageHeight;
}

static inline void gblSetImageHeight(int imageHeight)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.imageHeight = imageHeight;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline int gblGetYFactor(void)
{
    int yFactor;

    pthread_mutex_lock(&gbl.mutex);
    yFactor = gbl.yFactor;
    pthread_mutex_unlock(&gbl.mutex);

    return yFactor;
}

static inline void gblSetYFactor(int yFactor)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.yFactor = yFactor;
    pthread_mutex_unlock(&gbl.mutex);
}

static inline void gblResetDone(unsigned int doneFlag)
{
    pthread_mutex_lock(&gbl.mutex);
    gbl.doneMask = gbl.doneMask & (~doneFlag);
    if (gbl.doneMask == 0) {
        gbl.done = FALSE;
    }
    pthread_mutex_unlock(&gbl.mutex);
}

/* Returns true if all threads are done playing */
static inline int gblAllDone(unsigned int doneFlag)
{
    int done;

    pthread_mutex_lock(&gbl.mutex);
    gbl.doneMask |= doneFlag;
    if ((gbl.targetMask & gbl.doneMask) == gbl.targetMask) {
        gbl.done = TRUE;
    }
    done = gbl.done;
    pthread_mutex_unlock(&gbl.mutex);

    return done; 
}

/* Scale vertically depending on video standard */
#define yScale(x) (((x) * gblGetYFactor()) / 10)

/* Cleans up cleanly after a failure */
#define cleanup(x)                                  \
    status = (x);                                   \
    gblSetQuit();                                   \
    goto cleanup

/* Breaks a processing loop for a clean exit */
#define breakLoop(x)                                \
    status = (x);                                   \
    gblSetQuit();                                   \
    break

#endif /* _DECODE_H */
