//*****************************************************************************
//
// ektm4c129_qssi_bootloader.c - The main file to prototype QSPI Flash Functions.
//
// Copyright (c) 2013-2014 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
//
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
//
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
//
// This is part of revision 2.1.0.12573 of the EK-TM4C1294XL Firmware Package.
//
//*****************************************************************************
#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_epi.h"
#include "inc/hw_gpio.h"
#include "inc/hw_memmap.h"
#include "inc/hw_nvic.h"
#include "inc/hw_ssi.h"
#include "inc/hw_sysctl.h"
#include "inc/hw_types.h"
#include "inc/hw_uart.h"
#include "bl_config.h"
#include "ektm4c129_qssi_bootloader.h"

//*****************************************************************************
//
// Defines for GPIO Pins for QSSI1, EPI0, LED and USER-SW
//
//*****************************************************************************
#define EPI_PORTA_PINS  0xC0
#define EPI_PORTB_PINS  0x0C
#define EPI_PORTC_PINS  0xF0
#define EPI_PORTG_PINS  0x03
#define EPI_PORTH_PINS  0x0F
#define EPI_PORTK_PINS  0xF0
#define EPI_PORTL_PINS  0x3F
#define EPI_PORTM_PINS  0x0F
#define EPI_PORTN_PINS  0x30
#define EPI_PORTP_PINS  0x0C
#define EPI_PORTQ_PINS  0x0F

#define EPI_PORTA_PCTL  0xFF000000
#define EPI_PORTB_PCTL  0x0000FF00
#define EPI_PORTC_PCTL  0xFFFF0000
#define EPI_PORTG_PCTL  0x000000FF
#define EPI_PORTH_PCTL  0x0000FFFF
#define EPI_PORTK_PCTL  0xFFFF0000
#define EPI_PORTL_PCTL  0x00FFFFFF
#define EPI_PORTM_PCTL  0x0000FFFF
#define EPI_PORTN_PCTL  0x00FF0000
#define EPI_PORTP_PCTL  0x0000FF00
#define EPI_PORTQ_PCTL  0x0000FFFF

#define QSSI_CLK_PINS   0x20
#define QSSI_FSS_PINS   0x10
#define QSSI_XDAT0_PINS 0x10
#define QSSI_XDAT1_PINS 0x20

#define QSSI_CLK_PCTL   0x00F00000
#define QSSI_FSS_PCTL   0x000F0000
#define QSSI_XDAT0_PCTL 0x000F0000
#define QSSI_XDAT1_PCTL 0x00F00000

#define LED_DIAG0_PINS  0x01
#define LED_DIAG1_PINS  0x02

#define USER_SW1_PINS   0x01
#define USER_SW2_PINS   0x02

//*****************************************************************************
//
// Defines for QSPI Instruction and Fixed Data
//
//*****************************************************************************
#define QSPI_INST_REMS  0x90
#define QSPI_INST_READ  0x03
#define QSPI_INST_WREN  0x06
#define QSPI_INST_SE4K  0x20
#define QSPI_INST_RDSR  0x05
#define QSPI_INST_PP    0x02

#define QSPI_RDSR_WIP   0x01

#define QSPI_REMS_DATA  0xC219

//*****************************************************************************
//
// Defines for SDRAM Application Address
//
//*****************************************************************************
#define SDRAM_APP_START_ADDRESS 0x60000000

//*****************************************************************************
//
// A prototype for the function (in the startup code) for a predictable length
// delay.
//
//*****************************************************************************
extern void Delay(uint32_t ui32Count);

//*****************************************************************************
//
// Global Variable to hold the Transfer Address and Size
//
//*****************************************************************************
extern uint32_t g_ui32TransferAddress;
extern uint32_t g_ui32TransferSize;

//*****************************************************************************
//
//! Configures the IO's of LED Module for Diagnostics
//!
//! \return None.
//
//*****************************************************************************
void
ConfigureLEDIOs(void)
{
	//
	// Enable Diagnostic Port N for LED Blinking
	//
	HWREG(SYSCTL_RCGCGPIO) |= SYSCTL_RCGCGPIO_R12;

	//
	// Wait for the Clocks to be enabled for Port N module
	//
	while((HWREG(SYSCTL_PRGPIO) & SYSCTL_PRGPIO_R12) != SYSCTL_PRGPIO_R12);

	//
	// Configure PN0 and PN1 LED Pins for Diagnosis Message
	// and drive them to 0.
	//
	HWREG(GPIO_PORTN_BASE + GPIO_O_AFSEL)     &= ~(LED_DIAG0_PINS | LED_DIAG1_PINS);
	HWREG(GPIO_PORTN_BASE + GPIO_O_DIR)       |= (LED_DIAG0_PINS | LED_DIAG1_PINS);
	HWREG(GPIO_PORTN_BASE + GPIO_O_DEN)       |= (LED_DIAG0_PINS | LED_DIAG1_PINS);
	HWREG(GPIO_PORTN_BASE + ((LED_DIAG0_PINS | LED_DIAG1_PINS) << 2))
						= 0x0;
}

//*****************************************************************************
//
//! Configures the IO's of User Switch for Boot Mode Selection
//!
//! \return None.
//
//*****************************************************************************
void
ConfigureUserSwitchIOs(void)
{
	//
	// Enable Diagnostic Port J for User Switch
	//
	HWREG(SYSCTL_RCGCGPIO) |= SYSCTL_RCGCGPIO_R8;

	//
	// Wait for the Clocks to be enabled for Port J module
	//
	while((HWREG(SYSCTL_PRGPIO) & SYSCTL_PRGPIO_R8) != SYSCTL_PRGPIO_R8);

	//
	// Configure PJ0 for User Switch as Input with Pull Up
	//
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_AFSEL)     &= ~(USER_SW1_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_DIR)       &= ~(USER_SW1_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_PUR)       |= (USER_SW1_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_DEN)       |= (USER_SW1_PINS);

	//
	// Configure PJ1 for User Switch as Input with Pull Up
	//
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_AFSEL)     &= ~(USER_SW2_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_DIR)       &= ~(USER_SW2_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_PUR)       |= (USER_SW2_PINS);
	HWREG(GPIO_PORTJ_AHB_BASE + GPIO_O_DEN)       |= (USER_SW2_PINS);

}

//*****************************************************************************
//
//! Configures the IO's of QSSI Module.
//!
//! \return None.
//
//*****************************************************************************
void
ConfigureQSSIIOs(void)
{
	//
	// Enable GPIO Port B, D and E for QSSI
	//
	HWREG(SYSCTL_RCGCGPIO) |= (SYSCTL_RCGCGPIO_R1 | SYSCTL_RCGCGPIO_R3 |
			SYSCTL_RCGCGPIO_R4);

	//
	// Wait for the Clocks to be enabled for Port B, D, E module
	//
	while((HWREG(SYSCTL_PRGPIO) & (SYSCTL_RCGCGPIO_R1 | SYSCTL_RCGCGPIO_R3 |
			SYSCTL_RCGCGPIO_R4)) != (SYSCTL_RCGCGPIO_R1 | SYSCTL_RCGCGPIO_R3 |
					SYSCTL_RCGCGPIO_R4));

	//
	// Configure QSSI1 IO Pins for CLK, FSS, XDAT0 and XDAT1
	//
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_PCTL)  |= (QSSI_CLK_PCTL | QSSI_FSS_PCTL);
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_DEN)   |= (QSSI_CLK_PINS | QSSI_FSS_PINS);
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_DR8R)  |= (QSSI_CLK_PINS | QSSI_FSS_PINS);
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_AFSEL) |= (QSSI_CLK_PINS | QSSI_FSS_PINS);

	HWREG(GPIO_PORTE_AHB_BASE + GPIO_O_PCTL)  |= (QSSI_XDAT0_PCTL | QSSI_XDAT1_PCTL);
	HWREG(GPIO_PORTE_AHB_BASE + GPIO_O_DEN)   |= (QSSI_XDAT0_PINS | QSSI_XDAT1_PINS);
	HWREG(GPIO_PORTE_AHB_BASE + GPIO_O_DR8R)  |= (QSSI_XDAT0_PINS | QSSI_XDAT1_PINS);
	HWREG(GPIO_PORTE_AHB_BASE + GPIO_O_AFSEL) |= (QSSI_XDAT0_PINS | QSSI_XDAT1_PINS);
}

//*****************************************************************************
//
//! Configures the IO's of EPI Module.
//!
//! \return None.
//
//*****************************************************************************
void
ConfigureEPIIOs(void)
{
	//
	// Enable GPIO Port A, B, C, G, H, K, L, M, N, P and Q for EPI
	//
	HWREG(SYSCTL_RCGCGPIO) |= (SYSCTL_RCGCGPIO_R0 | SYSCTL_RCGCGPIO_R1 |
			SYSCTL_RCGCGPIO_R2 | SYSCTL_RCGCGPIO_R6 | SYSCTL_RCGCGPIO_R7 |
			SYSCTL_RCGCGPIO_R9 | SYSCTL_RCGCGPIO_R10 | SYSCTL_RCGCGPIO_R11 |
			SYSCTL_RCGCGPIO_R12 | SYSCTL_RCGCGPIO_R13 | SYSCTL_RCGCGPIO_R14);

	//
	// Wait for the Clocks to be enabled for Port A, B, C, G, H, K, L, M,
	// N, P and Q
	//
	while((HWREG(SYSCTL_PRGPIO) & (SYSCTL_PRGPIO_R0 | SYSCTL_PRGPIO_R1 |
			SYSCTL_PRGPIO_R2 | SYSCTL_PRGPIO_R6 | SYSCTL_PRGPIO_R7 |
			SYSCTL_PRGPIO_R9 | SYSCTL_PRGPIO_R10 | SYSCTL_PRGPIO_R11 |
			SYSCTL_PRGPIO_R12 | SYSCTL_PRGPIO_R13 | SYSCTL_PRGPIO_R14)) !=
					(SYSCTL_PRGPIO_R0 | SYSCTL_PRGPIO_R1 |
					SYSCTL_PRGPIO_R2 | SYSCTL_PRGPIO_R6 | SYSCTL_PRGPIO_R7 |
					SYSCTL_PRGPIO_R9 | SYSCTL_PRGPIO_R10 | SYSCTL_PRGPIO_R11 |
					SYSCTL_PRGPIO_R12 | SYSCTL_PRGPIO_R13 | SYSCTL_PRGPIO_R14));

	//
	// Configure EPI IO Pins
	//
	HWREG(GPIO_PORTA_AHB_BASE + GPIO_O_PCTL)  |= EPI_PORTA_PCTL;
	HWREG(GPIO_PORTA_AHB_BASE + GPIO_O_DEN)   |= EPI_PORTA_PINS;
	HWREG(GPIO_PORTA_AHB_BASE + GPIO_O_DR8R)  |= EPI_PORTA_PINS;
	HWREG(GPIO_PORTA_AHB_BASE + GPIO_O_AFSEL) |= EPI_PORTA_PINS;

	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_PCTL)  |= EPI_PORTB_PCTL;
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_DEN)   |= EPI_PORTB_PINS;
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_DR8R)  |= EPI_PORTB_PINS;
	HWREG(GPIO_PORTB_AHB_BASE + GPIO_O_AFSEL) |= EPI_PORTB_PINS;

	HWREG(GPIO_PORTC_AHB_BASE + GPIO_O_PCTL)  |= EPI_PORTC_PCTL;
	HWREG(GPIO_PORTC_AHB_BASE + GPIO_O_DEN)   |= EPI_PORTC_PINS;
	HWREG(GPIO_PORTC_AHB_BASE + GPIO_O_DR8R)  |= EPI_PORTC_PINS;
	HWREG(GPIO_PORTC_AHB_BASE + GPIO_O_AFSEL) |= EPI_PORTC_PINS;

	HWREG(GPIO_PORTG_AHB_BASE + GPIO_O_PCTL)  |= EPI_PORTG_PCTL;
	HWREG(GPIO_PORTG_AHB_BASE + GPIO_O_DEN)   |= EPI_PORTG_PINS;
	HWREG(GPIO_PORTG_AHB_BASE + GPIO_O_DR8R)  |= EPI_PORTG_PINS;
	HWREG(GPIO_PORTG_AHB_BASE + GPIO_O_AFSEL) |= EPI_PORTG_PINS;

	HWREG(GPIO_PORTH_AHB_BASE + GPIO_O_PCTL)  |= EPI_PORTH_PCTL;
	HWREG(GPIO_PORTH_AHB_BASE + GPIO_O_DEN)   |= EPI_PORTH_PINS;
	HWREG(GPIO_PORTH_AHB_BASE + GPIO_O_DR8R)  |= EPI_PORTH_PINS;
	HWREG(GPIO_PORTH_AHB_BASE + GPIO_O_AFSEL) |= EPI_PORTH_PINS;

	HWREG(GPIO_PORTK_BASE + GPIO_O_PCTL)  |= EPI_PORTK_PCTL;
	HWREG(GPIO_PORTK_BASE + GPIO_O_DEN)   |= EPI_PORTK_PINS;
	HWREG(GPIO_PORTK_BASE + GPIO_O_DR8R)  |= EPI_PORTK_PINS;
	HWREG(GPIO_PORTK_BASE + GPIO_O_AFSEL) |= EPI_PORTK_PINS;

	HWREG(GPIO_PORTL_BASE + GPIO_O_PCTL)  |= EPI_PORTL_PCTL;
	HWREG(GPIO_PORTL_BASE + GPIO_O_DEN)   |= EPI_PORTL_PINS;
	HWREG(GPIO_PORTL_BASE + GPIO_O_DR8R)  |= EPI_PORTL_PINS;
	HWREG(GPIO_PORTL_BASE + GPIO_O_AFSEL) |= EPI_PORTL_PINS;

	HWREG(GPIO_PORTM_BASE + GPIO_O_PCTL)  |= EPI_PORTM_PCTL;
	HWREG(GPIO_PORTM_BASE + GPIO_O_DEN)   |= EPI_PORTM_PINS;
	HWREG(GPIO_PORTM_BASE + GPIO_O_DR8R)  |= EPI_PORTM_PINS;
	HWREG(GPIO_PORTM_BASE + GPIO_O_AFSEL) |= EPI_PORTM_PINS;

	HWREG(GPIO_PORTN_BASE + GPIO_O_PCTL)  |= EPI_PORTN_PCTL;
	HWREG(GPIO_PORTN_BASE + GPIO_O_DEN)   |= EPI_PORTN_PINS;
	HWREG(GPIO_PORTN_BASE + GPIO_O_DR8R)  |= EPI_PORTN_PINS;
	HWREG(GPIO_PORTN_BASE + GPIO_O_AFSEL) |= EPI_PORTN_PINS;

	HWREG(GPIO_PORTP_BASE + GPIO_O_PCTL)  |= EPI_PORTP_PCTL;
	HWREG(GPIO_PORTP_BASE + GPIO_O_DEN)   |= EPI_PORTP_PINS;
	HWREG(GPIO_PORTP_BASE + GPIO_O_DR8R)  |= EPI_PORTP_PINS;
	HWREG(GPIO_PORTP_BASE + GPIO_O_AFSEL) |= EPI_PORTP_PINS;

	HWREG(GPIO_PORTQ_BASE + GPIO_O_PCTL)  |= EPI_PORTQ_PCTL;
	HWREG(GPIO_PORTQ_BASE + GPIO_O_DEN)   |= EPI_PORTQ_PINS;
	HWREG(GPIO_PORTQ_BASE + GPIO_O_DR8R)  |= EPI_PORTQ_PINS;
	HWREG(GPIO_PORTQ_BASE + GPIO_O_AFSEL) |= EPI_PORTQ_PINS;
}

//*****************************************************************************
//
//! Configures the EPI Interface for SDRAM and check if it accessible
//!
//! \return 0 for Failure of EPI SDRAM Init or 1 for Success.
//
//*****************************************************************************
uint32_t
ConfigureEPIInterface(void)
{
	uint32_t ui32Config;

	//
	// Disable and Reset the EPI0 Module at every start
	//
	HWREG(SYSCTL_RCGCEPI)  &= ~(SYSCTL_RCGCEPI_R0);
	HWREG(SYSCTL_SREPI)    |= (SYSCTL_RCGCEPI_R0);
	Delay(16);
	HWREG(SYSCTL_SREPI)    &= ~(SYSCTL_RCGCEPI_R0);

	//
	// Enable Clock to EPI0 Module
	//
	HWREG(SYSCTL_RCGCEPI)  |= (SYSCTL_RCGCEPI_R0);

	//
	// Wait for the Clocks to be enabled for EPI0 module
	// as it is the last of the modules to be enabled
	//
	while((HWREG(SYSCTL_PREPI) & SYSCTL_PREPI_R0) != SYSCTL_PREPI_R0);

	//
	// Set the Divider for Clock Same as the System Clock
	//
    HWREG(EPI0_BASE + EPI_O_BAUD) = 0x0;

    //
    // Write the mode word to the register for SDRAM Mode
    //
    HWREG(EPI0_BASE + EPI_O_CFG) = 0x00000011;

    //
    // Write the SDRAM configuration register.
    //
    HWREG(EPI0_BASE + EPI_O_SDRAMCFG) = (0x40000000 | 0x00000003);

    //
    // Set the value of the address mapping register.
    //
    ui32Config = (0x0000000C | 0x00000001);
    ui32Config &= ~EPI_SDRAMCFG_RFSH_M;
    ui32Config |= 125 << EPI_SDRAMCFG_RFSH_S;
    HWREG(EPI0_BASE + EPI_O_ADDRMAP) = ui32Config;

    //
    // Wait for the SDRAM wake-up to complete by polling the SDRAM
    // initialization sequence bit.  This bit is true when the SDRAM interface
    // is going through the initialization and false when the SDRAM interface
    // it is not in a wake-up period.
    //
    while(HWREG(EPI0_BASE + EPI_O_STAT) &  EPI_STAT_INITSEQ)
    {
    }

    HWREGH(SDRAM_APP_START_ADDRESS + 0x0) = 0xABCD;
    HWREGH(SDRAM_APP_START_ADDRESS + 0x2) = 0x1234;
    HWREGH(SDRAM_APP_START_ADDRESS + 0x1000) = 0xDCBA;
    HWREGH(SDRAM_APP_START_ADDRESS + 0x1002) = 0x4321;

    if((HWREG(SDRAM_APP_START_ADDRESS) == 0x1234ABCD) && (HWREG(SDRAM_APP_START_ADDRESS + 0x1000) == 0x4321DCBA))
    {
    	return(1);
    }
    else
    {
        return(0);
    }
}

//*****************************************************************************
//
//! Configures the QSSI Interface for QSPI Serial Flash and check if External
//! serial flash is accessible
//!
//! \return Device ID for Macronix Flash.
//
//*****************************************************************************
uint16_t
ConfigureQSSIInterface(void)
{
	uint16_t ui16DeviceID;

	//
	// Disable and Reset the QSSI1 Module at every start
	//
	HWREG(SYSCTL_RCGCSSI)  &= ~(SYSCTL_RCGCSSI_R1);
	HWREG(SYSCTL_SRSSI)    |= (SYSCTL_RCGCSSI_R1);
	Delay(16);
	HWREG(SYSCTL_SRSSI)    &= ~(SYSCTL_RCGCSSI_R1);

	//
	// Enable Clock to QSSI1 Module
	//
	HWREG(SYSCTL_RCGCSSI)  |= (SYSCTL_RCGCSSI_R1);

	//
	// Wait for the Clocks to be enabled for QSSI1 module
	// as it is the last of the modules to be enabled
	//
	while((HWREG(SYSCTL_PRSSI) & SYSCTL_PRSSI_R1) != SYSCTL_PRSSI_R1);

	//
	// Configure QSPI with Functional Clock set as PIOSC
	// and Serial Clock Set to half of PIOSC
	//
	HWREG(SSI1_BASE + SSI_O_CC)   |= SSI_CC_CS_PIOSC;
	HWREG(SSI1_BASE + SSI_O_CR0)  = (SSI_CR0_FRF_MOTO | SSI_CR0_DSS_8);
	HWREG(SSI1_BASE + SSI_O_CPSR) = 0x2;
	HWREG(SSI1_BASE + SSI_O_CR1)  = (SSI_CR1_HSCLKEN | SSI_CR1_SSE);

	//
	// Perform Advanced Device ID Read of QSPI
	//
	HWREG(SSI1_BASE + SSI_O_CR1)  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_REMS;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;

	HWREG(SSI1_BASE + SSI_O_CR1)  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED | SSI_CR1_DIR);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
	ui16DeviceID = (HWREG(SSI1_BASE + SSI_O_DR) << 8);

	HWREG(SSI1_BASE + SSI_O_CR1)  |= SSI_CR1_EOM;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
	ui16DeviceID |= HWREG(SSI1_BASE + SSI_O_DR);

	return(ui16DeviceID);
}

//*****************************************************************************
//
//! Reads the User Switch and returns the value
//!
//! \return Switch value of 0 (pressed) or 1 (unpressed).
//
//*****************************************************************************
uint32_t
ReadUserSwitch(uint32_t ui32Switch)
{
	return(HWREG(GPIO_PORTJ_AHB_BASE+(ui32Switch << 2)));
}

//*****************************************************************************
//
//! QSPI Serial Flash Read Function.
//!
//! \return the ui8NumberOfReadBytes bytes of read data in array pointed by
//! *ui8ReadData
//
//*****************************************************************************
void
QSPIReadFunction(uint32_t ui32Address, uint8_t *ui8ReadData, uint16_t ui16NumberOfReadBytes)
{
	uint32_t ui32Count;
	uint32_t ui32Config;

	//
	// Configure QSSI for Advanced Write Mode
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	//
	// Send Command for Read Instruction
	//
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_READ;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0xFF0000) >> 16);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x00FF00) >> 8);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x0000FF) >> 0);

	//
	// Configure QSSI for Advanced Read Mode to receive data
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_DIR | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	//
	// Begin Reading the data and putting it into the buffer
	//
	for(ui32Count = 0 ; ui32Count < (ui16NumberOfReadBytes-1) ; ui32Count++)
	{
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
		ui8ReadData[ui32Count] = HWREG(SSI1_BASE + SSI_O_DR);
	}

	//
	// For the last data send EOM
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_DIR | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = 0x00;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
	ui8ReadData[ui16NumberOfReadBytes-1] = HWREG(SSI1_BASE + SSI_O_DR);
}

//*****************************************************************************
//
//! QSPI Serial Flash Erase Function.
//!
//! \return None
//
//*****************************************************************************
void
QSPIEraseFunction(uint32_t ui32Address)
{
	uint8_t  ui8Status;
	uint32_t ui32Config;

	//
	// Configure QSSI for Advanced Write Mode
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	//
	// Send Write Enable Command
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_WREN;

	//
	// Wait Till SSI Is Busy
	//
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

	//
	// Send Erase Command for Erase Instruction in advanced write mode
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_SE4K;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0xFF0000) >> 16);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x00FF00) >> 8);

	//
	// For Address Bits 7-0 Set the EOM to terminate the transaction
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x0000FF) >> 0);

	//
	// Wait Till SSI Is Busy
	//
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

	//
	// Poll WIP Bit to Clear for Sector Erase to Complete
	//
	ui8Status = 0x0;

	do
	{
		//
		// Configure QSSI for Advanced Write Mode
		//
		ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
		ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
		ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
		HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

		//
		// Write the command for Read Status Register
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)  = QSPI_INST_RDSR;

		//
		// Change direction for Reading the Byte and write a dummy byte
		//
		ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
		ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_DIR | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
		HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)  = 0x00;

		//
		// Wait Till SSI Is Busy
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

		//
		// Read the Byte for SSI RXFIFO
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
		ui8Status = HWREG(SSI1_BASE + SSI_O_DR);
	} while((ui8Status & QSPI_RDSR_WIP) != 0x00);
}

//*****************************************************************************
//
//! QSPI Serial Flash Program Function.
//!
//! \return None
//
//*****************************************************************************
void
QSPIProgramFunction(uint32_t ui32Address, uint8_t *pucSrcData, uint32_t ui32Length)
{
	uint8_t  ui8Status;
	uint32_t ui32Loop;
	uint32_t ui32Config;

	//
	// Configure QSSI for Advanced Write Mode
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	//
	// Send Write Enable Command
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_WREN;

	//
	// Wait Till SSI Is Busy
	//
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

	//
	// Send Command for Program Instruction in advanced write mode
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = QSPI_INST_PP;
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0xFF0000) >> 16);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x00FF00) >> 8);
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = ((ui32Address & 0x0000FF) >> 0);

	//
	// Send Data for the address mentioned
	//
	for(ui32Loop = 0 ; ui32Loop < (ui32Length-1) ; ui32Loop++)
	{
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)   = pucSrcData[ui32Loop];
	}

	//
	// Send EOM With the last data
	//
	//
	// For Address Bits 7-0 Set the EOM to terminate the transaction
	//
	ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
	ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
	HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
	HWREG(SSI1_BASE + SSI_O_DR)   = pucSrcData[ui32Length-1];

	//
	// Wait Till SSI Is Busy
	//
	while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

	//
	// Poll WIP Bit to Clear for Sector Erase to Complete
	//
	ui8Status = 0x0;

	do
	{
		//
		// Configure QSSI for Advanced Write Mode
		//
		ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
		ui32Config  &= ~(SSI_CR1_EOM | SSI_CR1_DIR);
		ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_MODE_ADVANCED);
		HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

		//
		// Write the command for Read Status Register
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)  = QSPI_INST_RDSR;

		//
		// Change direction for Reading the Byte and write a dummy byte
		//
		ui32Config  = HWREG(SSI1_BASE + SSI_O_CR1);
		ui32Config  |= (SSI_CR1_FSSHLDFRM | SSI_CR1_DIR | SSI_CR1_EOM | SSI_CR1_MODE_ADVANCED);
		HWREG(SSI1_BASE + SSI_O_CR1) = ui32Config;

		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_TFE) != SSI_SR_TFE);
		HWREG(SSI1_BASE + SSI_O_DR)  = 0x00;

		//
		// Wait Till SSI Is Busy
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_BSY) == SSI_SR_BSY);

		//
		// Read the Byte for SSI RXFIFO
		//
		while((HWREG(SSI1_BASE + SSI_O_SR) & SSI_SR_RNE) != SSI_SR_RNE);
		ui8Status = HWREG(SSI1_BASE + SSI_O_DR);
	} while((ui8Status & QSPI_RDSR_WIP) != 0x00);

}

//*****************************************************************************
//
//! Reads the QSPI Flash Memory Sector 0 to check if there is an image
//| pointer. An Image pointer is qualifed as a 2 word filed
//! Word-0: upper 16-bits for Sector Start, one reserved byte and
//!         lower byte for valid image SSSSRRVV
//!         If VV == 0xFF then there is no image
//!         If VV == 0xFE then the image is a partial image at copy
//!         If VV == 0xFC then the image is a valid Image
//! Word-1: Size of Image in Bytes
//!
//! \return false if no valid image found. If valid image found then return
//! the sector number and ImageSize.
//
//*****************************************************************************
bool
GetQSPIFirmWareImagePointer(uint16_t *ui16SectorNumber, uint32_t *ui32ImageSize)
{
	uint8_t ui8ImagePointer[8];
	uint32_t ui32PointerAddress;

	//
	// Get the address of the Image Pointer Table
	//
	ui32PointerAddress = (0x1000 * 0);

	//
	// Read 2 Words of Pointer Header to locate the image
	//
	QSPIReadFunction(ui32PointerAddress, &ui8ImagePointer[0], 8);

	if(ui8ImagePointer[0] == 0xFF)
	{
		//
		// Return with no image found
		//
		return false;
	}
	else if(ui8ImagePointer[0] == 0xFC)

	{
		//
		// Return the Sector Number and Size of Imahe
		//
		*ui16SectorNumber = ui8ImagePointer[2] | (ui8ImagePointer[3] << 8);
		*ui32ImageSize    = ui8ImagePointer[4] | (ui8ImagePointer[5] << 8) | (ui8ImagePointer[6] << 16) | (ui8ImagePointer[7] << 24);
		return true;
	}
	else
	{
		//
		// Return with no image found and TODO: erase Serial
		// Flash
		//
		return false;
	}

}

//*****************************************************************************
//
//! Reads the QSPI pointed by Start Address to SDRAM till Size of Image
//!
//! \return None.
//
//*****************************************************************************
void
DownloadQSPItoSDRAM(uint16_t ui16SectorNumber, uint32_t ui32ImageSize)
{
	uint8_t  ui8ApplicationImage[256];
	uint32_t ui32SerialImageAddress;
	uint32_t ui32SDRAMImageAddress;
	uint32_t ui32Count;

	//
	// Initialize SDRAM Address
	//
	ui32SDRAMImageAddress = 0;

	//
	// First get the Start Address of the Image in QSPI
	// serial Flash Memory.
	//
	ui32SerialImageAddress = (0x1000 * ui16SectorNumber);


	//
	// Read 256 bytes of Image at a time and then program the same
	//
	while((ui32SerialImageAddress - (0x1000 * ui16SectorNumber)) < ui32ImageSize)
	{
		QSPIReadFunction(ui32SerialImageAddress, &ui8ApplicationImage[0], 256);
		for(ui32Count = 0 ; ui32Count < 256 ; ui32Count++)
		{
			HWREGB(SDRAM_APP_START_ADDRESS + ui32SDRAMImageAddress) = ui8ApplicationImage[ui32Count];
			ui32SDRAMImageAddress++;
		}
		ui32SerialImageAddress += 256;
	}

}

//*****************************************************************************
//
// This function is used to call the user application.  It will set the NVIC
// to point at the user app's vector table, load up the user app's stack
// pointer, and then jump to the application.
//
// This function must be programmed in assembly since it needs to directly
// manipulate the value in the stack pointer, and because it needs to perform
// a direct branch to the user app and not a function call (bl).
//
//*****************************************************************************
#if defined(codered) || defined(gcc) || defined(sourcerygxx)
void __attribute__((naked))
CallApplication(uint_fast32_t ui32StartAddr)
{
    //
    // Set the vector table to the beginning of the app in flash.
    //
    HWREG(NVIC_VTABLE) = ui32StartAddr;

    //
    // Load the stack pointer from the application's vector table.
    //
    __asm("    ldr     r1, [r0]\n"
          "    mov     sp, r1");

    //
    // Load the initial PC from the application's vector table and branch to
    // the application's entry point.
    //
    __asm("    ldr     r0, [r0, #4]\n"
          "    bx      r0\n");
}
#elif defined(ewarm)
void
CallApplication(uint_fast32_t ui32StartAddr)
{
    //
    // Set the vector table to the beginning of the app in flash.
    //
    HWREG(NVIC_VTABLE) = ui32StartAddr;

    //
    // Load the stack pointer from the application's vector table.
    //
    __asm("    ldr     r1, [r0]\n"
          "    mov     sp, r1");

    //
    // Load the initial PC from the application's vector table and branch to
    // the application's entry point.
    //
    __asm("    ldr     r0, [r0, #4]\n"
          "    bx      r0\n");
}
#elif defined(rvmdk) || defined(__ARMCC_VERSION)
__asm void
CallApplication(uint_fast32_t ui32StartAddr)
{
    //
    // Set the vector table address to the beginning of the application.
    //
    ldr     r1, =0xe000ed08
    str     r0, [r1]

    //
    // Load the stack pointer from the application's vector table.
    //
    ldr     r1, [r0]
    mov     sp, r1

    //
    // Load the initial PC from the application's vector table and branch to
    // the application's entry point.
    //
    ldr     r0, [r0, #4]
    bx      r0
}
#elif defined(ccs)
void
CallApplication(uint_fast32_t ui32StartAddr)
{
    //
    // Set the vector table to the beginning of the app in flash.
    //
    HWREG(NVIC_VTABLE) = ui32StartAddr;

#if 0
    __asm("    movw r0, #0x0000\n"
          "    movt r0, #0x6000\n"
          "    ldr  sp, [r0]\n"
          "    ldr  r0, [r0, #4]\n"
          "    bx   r0");
#endif
    //
    // Load the stack pointer from the application's vector table.
    // Load the initial PC from the application's vector table and branch to
    // the application's entry point.
    //
    __asm("    ldr     sp, [r0]\n"
          "    ldr     r0, [r0, #4]\n"
          "    bx      r0\n");
}
#else
#error Undefined compiler!
#endif

//*****************************************************************************
//
//! Configures the QSSI Interface for Serial Flash.
//!
//! \return None.
//
//*****************************************************************************
void QSPIInitFunction(void)
{
	uint16_t ui16StartSectorNumber;
	uint32_t ui32ImageSizeInBytes;
	bool bStatus;

	//
	// Initialize Return Status
	//
	bStatus = false;

	//
	// Configure IO's for LED, User Switch, QSSI and EPI
	//
	ConfigureLEDIOs();
	ConfigureUserSwitchIOs();
	ConfigureQSSIIOs();
	ConfigureEPIIOs();

	//
	// Check if QSPI Flash Serial ID checks out as QSPI_REMS_DATA
	// If not, then hang in the loop with D1 Blinking
	//
	if(ConfigureQSSIInterface() != QSPI_REMS_DATA)
	{
		while(1)
		{
			Delay(16000000/9);
			HWREG(GPIO_PORTN_BASE + ((LED_DIAG0_PINS | LED_DIAG1_PINS) << 2))
									^= (LED_DIAG1_PINS);
		}
	}

	//
	// Check if EPI SDRAM is available
	// If not, then hang in the loop with D2 Blinking
	//
	if(ConfigureEPIInterface() != 0x1)
	{
		while(1)
		{
			Delay(16000000/9);
			HWREG(GPIO_PORTN_BASE + ((LED_DIAG0_PINS | LED_DIAG1_PINS) << 2))
									^= (LED_DIAG0_PINS);
		}
	}

	//
	// Read the User Switch SW1. If the Switch is Low exit the function
	// to go to the Serial Boot Loader to download a firmware. Before
	// doing loader erase the Firmware Entry Table
	// If Switch is high then go and check the QSPI Serial Flash
	//
	if(ReadUserSwitch(USER_SW1_PINS))
	{
		bStatus = GetQSPIFirmWareImagePointer(&ui16StartSectorNumber, &ui32ImageSizeInBytes);

		//
		// No Image Found so go to main boot loader to download
		// a new firmware image
		if(!bStatus)
		{
			return;
		}

		//
		// If a valid Pointer is available then download the iamge to
		// SDRAM and jump to it for execution
		//
		DownloadQSPItoSDRAM(ui16StartSectorNumber, ui32ImageSizeInBytes);

		CallApplication(SDRAM_APP_START_ADDRESS);
	}
	else
	{
		//
		// Erase the Sector for corresponding Fimrware Entry
		// Page Table
		//
		QSPIEraseFunction(0);

		return;
	}

	while(1)
	{
		Delay(16000000/9);
		HWREG(GPIO_PORTN_BASE + ((LED_DIAG0_PINS | LED_DIAG1_PINS) << 2))
								= (LED_DIAG1_PINS | LED_DIAG0_PINS);
		Delay(16000000/9);
		HWREG(GPIO_PORTN_BASE + ((LED_DIAG0_PINS | LED_DIAG1_PINS) << 2))
								^= (LED_DIAG1_PINS | LED_DIAG0_PINS);
	}

	// Configure USR_SW1 and check for boot loader service request
		// If yes then Erase QSPI Flash and proceed to boot loader
		// If not then Check for Valid Image in QSPI
	// Check fo Valid Image in QSPI
		// If yes then copy to SDRAM and jump to SDRAM execution
		// If not then 1st b'day and proceed to boot loader

}

//*****************************************************************************
//
//! Sets the Marker for Valid Image
//!
//! \return 1.
//
//*****************************************************************************
void QSPIFlashEnd(void)
{
	uint8_t ui8ImagePointer[8];
	uint32_t ui32PointerAddress;

	//
	// Get the address of the Image Pointer Table
	//
	ui32PointerAddress = (0x1000 * 0);

	//
	// Read 2 Words of Pointer Header to locate the image
	//
	QSPIReadFunction(ui32PointerAddress, &ui8ImagePointer[0], 8);

	//
	// Change Marker to 0xFC
	//
	ui8ImagePointer[0] = 0xFC;

	//
	// Program the Sector Address and Size along with the
	// marker for copy
	//
	QSPIProgramFunction(ui32PointerAddress, &ui8ImagePointer[0], 8);
}

//*****************************************************************************
//
//! Programs the Address and Size of the Image to be downloaded
//! and the marker indicating a copy is being done.
//!
//! \return 1.
//
//*****************************************************************************
uint32_t
QSPIFlashAddressCheck(uint32_t ulAddr, uint32_t ulSize)
{
	uint8_t ui8ImagePointer[8];
	uint32_t ui32PointerAddress;
	uint32_t ui32SectorAddress;

	//
	// Format the data
	//
	ui32SectorAddress  = (ulAddr & ((uint32_t) ~(0xFFF))) >> 12;
	ui8ImagePointer[0] = 0xFE;
	ui8ImagePointer[1] = 0xFF;
	ui8ImagePointer[2] = (ui32SectorAddress & 0xFF);
	ui8ImagePointer[3] = ((ui32SectorAddress & 0xFF00) >> 8);
	ui8ImagePointer[4] = ((ulSize & 0x000000FF) >> 0);
	ui8ImagePointer[5] = ((ulSize & 0x0000FF00) >> 8);
	ui8ImagePointer[6] = ((ulSize & 0x00FF0000) >> 16);
	ui8ImagePointer[7] = ((ulSize & 0xFF000000) >> 24);


	//
	// Get the address of the Image Pointer Table
	//
	ui32PointerAddress = (0x1000 * 0);

	//
	// Program the Sector Address and Size along with the
	// marker for copy
	//
	QSPIProgramFunction(ui32PointerAddress, &ui8ImagePointer[0], 8);

	//
	// Return Success
	//
	return 1;
}

//*****************************************************************************
//
//! Clears Error Flags
//!
//! \return 1.
//
//*****************************************************************************
void
QSPIFlashClearError(void)
{

}

//*****************************************************************************
//
//! Checks if there is any error flag set
//!
//! \return 0.
//
//*****************************************************************************
uint32_t
QSPIFlashError(void)
{
	return 0;
}

//*****************************************************************************
//
//! Returns the size of the Serial Flash
//!
//! \return 0xFFFFFFFF
//
//*****************************************************************************
uint32_t
QSPIFlashSize(void)
{
	return (0xFFFFFFFF);
}

//*****************************************************************************
//
//! Erases the 4KB Sector in QSPI Serial Flash
//!
//! \return none.
//
//*****************************************************************************
void
QSPIFlashErase(uint32_t ulBlockAddr)
{
	//
	// Erase the Sector for corresponding memory pages
	//
	QSPIEraseFunction(ulBlockAddr & ((uint32_t) ~(0xFFF)));

	//
	// Wait for Some time before erasing
	//
	Delay(1000);

	//
	// Erase the Sector for corresponding memory pages
	//
	QSPIEraseFunction(ulBlockAddr & ((uint32_t) ~(0xFFF)));
}

//*****************************************************************************
//
//! Programs the Image to QSPI Serial Flash
//!
//! \return none.
//
//*****************************************************************************
void QSPIFlashProgram(uint32_t ulDstAddr, uint8_t *pucSrcData, uint32_t ulLength)
{
	//
	// Program the Bytes as given by the Array and Lenght at address
	// specified (address is 24-bit)
	//
	QSPIProgramFunction(ulDstAddr, pucSrcData, ulLength);

	//
	// Wait for Some time before returning to the main boot loader
	//
	Delay(1000);
}
