/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
//         Dual Ray Smoke Sensor GUI HAL for MSP430FR2355 using UART
//
// Driver to send and receive data from GUI using FR2355 UART
// Texas Instruments, Inc.
// Luis R
// *****************************************************************************
#include <DualRaySmokeAFE_GUI.h>
#include <DualRaySmokeAFE_HAL.h>
#include <DualRaySmokeAFE_Config_Common.h>
#include "DualRaySmokeAFE_HAL_Config_Private.h"

tGUICommRXCharCallback RxByteISRCallback;

void DualRaySmokeAFE_HAL_GUI_Init(tGUICommRXCharCallback RxByteCallback)
{
    // Store callback for ISR RX Byte
    RxByteISRCallback = RxByteCallback;

    // Configure UART for FR2355
    UCA1CTLW0 |= UCSWRST;
    UCA1CTLW0 |= UCSSEL__SMCLK;

#if ((DUALRAYSMOKEAFE_HAL_GUICOMM_BAUDRATE == 9600) && (HSBUS_FREQ_MHZ == 1))
    // Baud Rate calculation
    // 1000000/(16*9600) = 6.51
    // Fractional portion = 0.51
    // User's Guide Table 17-4: UCBRSx = 0x20
    // UCBRFx = 8
    UCA1BR0 = 6;
    UCA1BR1 = 0x00;
    UCA1MCTLW = 0x2000 | UCOS16 | UCBRF_8;
#elif ((DUALRAYSMOKEAFE_HAL_GUICOMM_BAUDRATE == 28800) && (HSBUS_FREQ_MHZ == 8))
    // Baud Rate calculation
    // 8E6/(16*28800) = 17.36
    // Fractional portion = 0.36
    // From eUSCI Calculator: UCBRSx = 221 (0xDD)
    // UCBRFx = 5
    UCA1BR0 = 17;
    UCA1BR1 = 0x00;
    UCA1MCTLW = 0xDD | UCOS16 | UCBRF_5;
#elif ((DUALRAYSMOKEAFE_HAL_GUICOMM_BAUDRATE == 19200) && (HSBUS_FREQ_MHZ == 8))
    // Baud Rate calculation
    // 8E6/(16*19200) = 26.04
    // Fractional portion = 0.04
    // User's Guide Table 17-4: UCBRSx = 0xB6
    // UCBRFx = 0
    UCA1BR0 = 26;
    UCA1BR1 = 0x00;
    UCA1MCTLW = 0xB6 | UCOS16 | UCBRF_0;
#elif ((DUALRAYSMOKEAFE_HAL_GUICOMM_BAUDRATE == 9600) && (HSBUS_FREQ_MHZ == 8))
    // Baud Rate calculation
    // 8E6/(16*9600) = 52.0833
    // Fractional portion = 0.33
    // User's Guide Table 17-4: UCBRSx = 0x49
    // UCBRFx = 1
    UCA1BR0 = 52;
    UCA1BR1 = 0x00;
    UCA1MCTLW = 0x49 | UCOS16 | UCBRF_1;
#else
#error "Define UART baudrate registers based on desired frequency"
#endif

    UCA1CTLW0 &= ~UCSWRST;                    // Initialize eUSCI
    UCA1IE |= UCRXIE;
}

void DualRaySmokeAFE_HAL_GUI_TransmitCharBlocking(char character)
{
    // Transmit Character
    while (UCA1STATW & UCBUSY)
        ;
    while (!(UCA1IFG & UCTXIFG))
        ;
    UCA1TXBUF = character;
    while (UCA1STATW & UCBUSY)
        ;
}


// EUSCI interrupt service routine
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_A1_VECTOR))) USCI_A1_ISR (void)
#else
#error Compiler not supported!
#endif
{
    switch(__even_in_range(UCA1IV,USCI_UART_UCTXCPTIFG))
    {
        case USCI_NONE: break;
        case USCI_UART_UCRXIFG:
            if (RxByteISRCallback != NULL)
            {
                 if (RxByteISRCallback(UCA1RXBUF) == true)
                 {
                     __bic_SR_register_on_exit(LPM4_bits);   // Exit LPM
                 }
            }
            break;
        case USCI_UART_UCTXIFG: break;
        case USCI_UART_UCSTTIFG: break;
        case USCI_UART_UCTXCPTIFG: break;
        default: break;
    }
}

