`timescale 1 ns / 100 ps
//*****************************************************************************
// (c) Copyright 2012, Texas Instruments                      
//      All Rights Reserved
//*****************************************************************************
// TEXAS INSTRUMENTS CONFIDENTIAL
//*****************************************************************************
// Module Name:  adc07d1520
//
// Description:  Top level module for the ADC07D1520 Reference Board
//
//*****************************************************************************
module adc07d1520
 (
  //cypress stuff
  input [15:0] cy_addr,
  inout [7:0]  cy_data,
  output [15:0] cy_fd,
  input fpga_reset,
  input clk100_p,
  input cs_n,
  input we_n,
  input rd_n,
  
  input flaga,
  input flagb,
  input flagc,
  input cy_req,
  input load_state,
  input ifclk,
  input [3:0] state,
  input [5:3] cy_ctl,
  output cy_valid,
      	      
  input slcs_n,
  input slrd,
  input slwr,
  input sloe,
  input pktend,
  output [3:0] cy_id,
  output [1:0] fifoadr,
  
  //adc clock, data, etc.
  input dclkiP,
  input dclkiN,
  input dclkqP,
  input dclkqN,
  input [11:0] dqP,
  input [11:0] dqN,
  input [11:0] dqdP,
  input [11:0] dqdN,
  input [11:0] diP,
  input [11:0] diN,
  input [11:0] didP,
  input [11:0] didN,
  input oriP,
  input oriN,
  input orqP,
  input orqN,
  input i_RCOutP1,
  input i_RCOutN1,
  input i_RCOutP2,
  input i_RCOutN2,
  
  
  //adc controls
  output pdi,
  output pdq,
  input  ece,
  output des,
  output fsr,
  output ddrphase,
  output caldly,tpm,
  output scs_adc,
  output sclk_adc,
  output ndm,
  output wss,
  output cal,
  input  adc_pwr,
  output poren,
  output sdi_adc,
  input sdo_adc,
  input calfail,
  input calrun,
      	      
  input ext_trigP,
  input ext_trigN,
  output [4:1] testp,testn,
  output [5:1] jp_,
  input [3:1] tcrit,
      	      
  //misc
  input pll_ld,
  output ext_clk_sel,
  output pll_le,pll_ce,
  output pll_sdata,
  output pll_sclk,
  output led_needcal,
  output led_overtemp,
  output led_standby,
  output led_acquire,
  output led_trigger,
  output led_overrange_i,
  output led_overrange_q,
  output led_health1,
  output led_health2,

  //usi spi
  input sdi_usi,
  output sdo_usi,
  output sclk_usi,
  output scs1_usi,
  output scs2_usi,

  //FMC Connector signals
  output o_RCOutP1,
  output o_RCOutN1,
  output o_RCOutP2,
  output o_RCOutN2,
  output o_FMC_PG_M2C,
  output o_FMC_PG_C2M,
  output o_CalRun,

  output o_DCLKIDP,
  output o_DCLKIDN,
  output o_DCLKQDP,
  output o_DCLKQDN,
  output o_DCLKIP,
  output o_DCLKIN,
  output o_DCLKQP,
  output o_DCLKQN,

  output o_ORIP,
  output o_ORIN,
  output o_ORQP,
  output o_ORQN,
  output [11:0] o_DIP,
  output [11:0] o_DIN,
  output [11:0] o_DIdP,
  output [11:0] o_DIdN,
  output [11:0] o_DQP,
  output [11:0] o_DQN,
  output [11:0] o_DQdP,
  output [11:0] o_DQdN
);
   
  wire i_rcout1, i_rcout2, i_rclk1, i_rclk2;
  wire rcout1_locked, rcout2_locked;
  wire w_dclki_p, w_dclki_n, w_dclkid_p, w_dclkid_n;
  wire w_dclkq_p, w_dclkq_n, w_dclkqd_p, w_dclkqd_n;
  wire w_rcout1_p, w_rcout1_n, w_rcout2_p, w_rcout2_n;
  wire w_ori_p, w_ori_n, w_orq_p, w_orq_n;
  wire [11:0] w_di_p, w_di_n, w_did_p, w_did_n;
  wire [11:0] w_dq_p, w_dq_n, w_dqd_p, w_dqd_n;
  

  assign o_RCOutP1     = w_rcout1_p;
  assign o_RCOutN1     = w_rcout1_n;
  assign o_RCOutP2     = w_rcout2_p;
  assign o_RCOutN2     = w_rcout2_n;
  assign o_FMC_PG_M2C  = 1'b1;
  assign o_FMC_PG_C2M  = 1'b1;
  assign o_CalRun      = calrun;

  assign o_DCLKIDP     = w_dclkid_p;
  assign o_DCLKIDN     = w_dclkid_n;
  assign o_DCLKQDP     = w_dclkqd_p;
  assign o_DCLKQDN     = w_dclkqd_n;
  assign o_DCLKIP      = w_dclki_p;
  assign o_DCLKIN      = w_dclki_n;
  assign o_DCLKQP      = w_dclkq_p;
  assign o_DCLKQN      = w_dclkq_n;

  assign o_ORIP	       = w_ori_p;
  assign o_ORIN	       = w_ori_n;
  assign o_ORQP	       = w_orq_p;
  assign o_ORQN	       = w_orq_n;

  assign o_DIP	       = w_di_p;
  assign o_DIN	       = w_di_n;
  assign o_DIdP	       = w_did_p;
  assign o_DIdN	       = w_did_n;
  assign o_DQP	       = w_dq_p;
  assign o_DQN	       = w_dq_n;
  assign o_DQdP	       = w_dqd_p;
  assign o_DQdN        = w_dqd_n;

//------------------------- Parameters -------------------------//
parameter SPEC_VER = 8'h15;
parameter FPGA_REV_NUM = 8'h01;

parameter [2:0]          //dcm reset state machine
  SM_DCMR_IDL  = 3'h0,
  SM_DCMR_ST1  = 3'h1,
  SM_DCMR_ST1A = 3'h2,
  SM_DCMR_ST1B = 3'h3,
  SM_DCMR_ST1C = 3'h4,
  SM_DCMR_ST2  = 3'h5;

parameter 
  SM_CAPT_IDL    = 3'h0,  //capture state machine
  SM_CAPT_IDL1   = 3'h1,  //capture state machine
  SM_CAPT_IDL2   = 3'h2,  //capture state machine
  SM_CAPT_ST1    = 3'h3,
  SM_CAPT_ST2    = 3'h4;

parameter 
  SM_FFR_IDL    = 3'h0,  //fifo read
  SM_FFR_ST1    = 3'h1,
  SM_FFR_ST2    = 3'h2,
  SM_FFR_ST3    = 3'h3;

parameter 		//Write Pulse state machine
  SM_WP_IDL    = 1'b0,
  SM_WP_ST1    = 1'b1;

parameter       //microwire state machine
  SM_MW_IDL    = 3'h0,
  SM_MW_ST0    = 3'h1,
  SM_MW_ST1    = 3'h2,
  SM_MW_ST2    = 3'h3,
  SM_MW_ST3    = 3'h4,
  SM_MW_ST4    = 3'h5,
  SM_MW_ST5    = 3'h6;

parameter       //lock state machine
  SM_HL_IDL    = 3'h0,
  SM_HL_ST0    = 3'h1,
  SM_HL_ST1    = 3'h2,
  SM_HL_ST2    = 3'h3,
  SM_HL_ST3    = 3'h4,
  SM_HL_ST4    = 3'h5;

//------------------------- Wires, Regs etc -------------------------//
    // clock nets
   wire  dclk_ib;
   wire  dclk_qb;
   wire  dclk_i;
   wire  dclk_q;
   wire  hdclk_i;
   wire  hdclk_q;
   wire  dcmi_lckd;
   wire  dcmq_lckd;
   wire  clk100;
   wire  clk6_25;
   wire  dcm100_lckd;

   // adc data nets from the IDDR primitives
   wire [11:0] di_r;
   wire [11:0] di_f;
   wire [11:0] did_r;
   wire [11:0] did_f;
   wire [11:0] dq_r;
   wire [11:0] dq_f;
   wire [11:0] dqd_r;
   wire [11:0] dqd_f;

   // first level of adc data registers
   reg [23:0]  di_reg;
   reg [23:0]  did_reg;
   reg [23:0]  dq_reg;
   reg [23:0]  dqd_reg;

   // second level of adc data registers (demuxing at half clock rate)
   reg [47:0]  di1_reg;
   reg [47:0]  did1_reg;
   reg [47:0]  dq1_reg;
   reg [47:0]  dqd1_reg;

   // third level of adc data registers (re-registering the data bus once more)
   reg [47:0]  di2_reg;
   reg [47:0]  did2_reg;
   reg [47:0]  dq2_reg;
   reg [47:0]  dqd2_reg;

   reg 	       demux_wen;  // toggles register enable for half clock rate data demux

   reg 	       fifo_wen;
   reg 	       fifo_wen_d1;
   reg 	       fifo_wr_en;
   
   reg 	       fifo_rst;
   reg 	       fifo_rst_d1;
   reg 	       fifo_reset;
   

   reg 	       cs_,wr_,rd_;
   reg [7:0]   cy_dt;
   reg [15:0]  cy_adr;

   reg 	       wp,sm_wp;
   reg [7:0]   cy_dato;

   reg [7:0]   dt_data_0,dt_data_1,dt_data_2,dt_data_3;
   wire [7:0]  tst_rg,dt_addr0,dt_addr1,dt_addr2;
   wire [7:2]  gl_csr;
   wire [7:0]  dt_pcnt0,dt_pcnt1,dt_cfg;
   wire [7:1]  dt_csr;
   wire [4:1]  spi_mctl;
   wire [7:0]  spi_cfg;

   wire [7:1]  adc_opt1;
   wire [2:0]  adc_opt21,adc_opt22;
   wire [4:0]  adc_opt3;
/*
  wire [31:0]  pll_reg0, pll_reg1, pll_reg2, pll_reg3, pll_reg4, pll_reg5;
  wire [31:0]  pll_reg6, pll_reg7, pll_reg8, pll_reg9, pll_reg10, pll_reg11;
*/
   reg 	       adcdcm_rst;
   reg [2:0]   sm_dcmr;

   wire        rr_tc;
   reg 	       srd,srd1,srd2,srd3,rr_tc1;
   reg 	       csr_strt,clr_csr_strt; //start bit in csr
   reg [16:0]  sr_cntr; //sample rate
   reg [15:0]  rr_cntr; //reference rate 

   reg [20:0]  clk_led_cntr;


   wire        cyaddr_vld = (cy_adr[15:8] == 8'hd0) && ~cy_adr[7];
   wire        rst = fpga_reset;

   reg [2:0]   sm_capt;
   reg 	       fifo_reg,fifo_ren;
  reg 	       fifo_ren_i, fifo_ren_q;

   reg [1:0]   sm_ffr;
   reg [2:0]   fcyc;

   wire [47:0] di_sync;
   wire [47:0] did_sync;
   wire [47:0] dq_sync;
   wire [47:0] dqd_sync;

   wire [95:0] fifo_i_in;
   wire [95:0] fifo_q_in;
   
   wire [95:0] fifo_i;
   wire [95:0] fifo_q;

   wire        fifo_i_mt;      // empty flas sync'ed to the clk100 domain (fifo read clock)
   wire        fifo_q_mt;      // empty flas sync'ed to the clk100 domain (fifo read clock)
   wire        fifo_i_full;    // full flag sync'ed to the hdclk_i domain (fifo write clock)
   wire        fifo_q_full;    // full flag sync'ed to the hdclk_i domain (fifo write clock)
   reg 	       fifo_i_full_d1; // first level of re-sync flops
   reg 	       fifo_q_full_d1; // first level of re-sync flops
   reg 	       fifo_i_full_d2; // second level of re-sync flops
   reg 	       fifo_q_full_d2; // second level of re-sync flops
   reg 	       fifo_full;      // fifo full flag sync'ed to teh clk100 domain

   reg [95:0]  fifo_mux;

   wire        ori,orq;

   wire        hdclk_actv_mx,hdclk_actv;
  wire 	       hdclk_actv_lckd;
  
   wire        iq_capt = adc_opt3[4];
   wire        hwtrg_en = adc_opt3[3];
   wire        q_ch     = adc_opt3[2];
   wire        des_mode = adc_opt3[1];

   wire        hw_trig;
   wire        hw_trgd  = ~hwtrg_en | (hw_trig & hwtrg_en);

reg [2:0] sm_hlck;
reg sm_pdq,clk_led_trn,clk_ledd,trg_hlck,clr_trg_hlck;

//////////////////////////////////////////////////
// SIGNALS for FMC Connector
//////////////////////////////////////////////////
  
 
  IBUFGDS #(.IOSTANDARD ("LVDS_25"), .DIFF_TERM ("TRUE"))
    i_rcout_1_bufgds
    (.I  (i_RCOutP1), 
     .IB (i_RCOutN1), 
     .O  (i_rcout1)
    );

  dcm_dclk dcm_rclk1 
   (.CLKIN_IN   (i_rcout1), 
    .RST_IN     (adcdcm_rst), 
    .CLKDV_OUT  (),
    .CLK0_OUT   (i_rclk1),
    .LOCKED_OUT (rcout1_locked)
   );

  IBUFGDS #(.IOSTANDARD ("LVDS_25"), .DIFF_TERM ("TRUE"))
    i_rcout_2_bufgds
    (.I  (i_RCOutP2), 
     .IB (i_RCOutN2), 
     .O  (i_rcout2)
    );

  dcm_dclk dcm_rclk2 
   (.CLKIN_IN   (i_rcout2), 
    .RST_IN     (adcdcm_rst), 
    .CLKDV_OUT  (),
    .CLK0_OUT   (i_rclk2),
    .LOCKED_OUT (rcout2_locked)
   );

  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_dclki_bufds
     (
      .I  (dclk_i),
      .O  (w_dclki_p),
      .OB (w_dclki_n)
     );
    
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_dclkq_bufds
     (
      .I  (dclk_q),
      .O  (w_dclkq_p),
      .OB (w_dclkq_n)
     );
  
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_dclkid_bufds
     (
      .I  (dclk_i),
      .O  (w_dclkid_p),
      .OB (w_dclkid_n)
     );
    
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_dclkqd_bufds
     (
      .I  (dclk_q),
      .O  (w_dclkqd_p),
      .OB (w_dclkqd_n)
     );
  
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_rcout1_bufds
     (
      .I  (i_rclk1),
      .O  (w_rcout1_p),
      .OB (w_rcout1_n)
     );
    
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_rcout2_bufds
     (
      .I  (i_rclk2),
      .O  (w_rcout2_p),
      .OB (w_rcout2_n)
     );
  
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_ori_bufds
     (
      .I  (ori),
      .O  (w_ori_p),
      .OB (w_ori_n)
     );
  
  OBUFDS #(.IOSTANDARD ("LVDS_25"))
    o_orq_bufds
     (
      .I  (ori),
      .O  (w_orq_p),
      .OB (w_orq_n)
     );
  
  adc1k_if_oddr adc_oddr_di
   (
    .dclk    (dclk_i), 
    .Dpin_N  (w_di_n), 
    .Dpin_P  (w_di_p), 
    .reset   (fpga_reset), 
    .SET_IN  (1'b0),
    .Din_1_r (di_r), 
    .Din_2_r (di_f)
   );

  adc1k_if_oddr adc_oddr_dq
   (
    .dclk    (dclk_i), 
    .Dpin_N  (w_dq_n), 
    .Dpin_P  (w_dq_p), 
    .reset   (fpga_reset), 
    .SET_IN  (1'b0),
    .Din_1_r (dq_r), 
    .Din_2_r (dq_f)
   );

  adc1k_if_oddr adc_oddr_did
   (
    .dclk    (dclk_i), 
    .Dpin_N  (w_did_n), 
    .Dpin_P  (w_did_p), 
    .reset   (fpga_reset), 
    .SET_IN  (1'b0),
    .Din_1_r (did_r), 
    .Din_2_r (did_f)
   );

  adc1k_if_oddr adc_oddr_dqd
   (
    .dclk    (dclk_i), 
    .Dpin_N  (w_dqd_n), 
    .Dpin_P  (w_dqd_p), 
    .reset   (fpga_reset), 
    .SET_IN  (1'b0),
    .Din_1_r (dqd_r), 
    .Din_2_r (dqd_f)
   );

//------------------------- Instantiations -------------------------//
IBUFGDS i_dclki_bufgds (
	                .I(dclkiP), 
                        .IB(dclkiN), 
                        .O(dclk_ib)
	                );
defparam i_dclki_bufgds.IOSTANDARD = "LVDS_25";
defparam i_dclki_bufgds.DIFF_TERM  = "TRUE";

dcm_dclk dcm_dclki (
	            .CLKIN_IN(dclk_ib), 
                    .RST_IN(adcdcm_rst), 
		    .CLKDV_OUT(hdclk_i),
                    .CLK0_OUT(dclk_i),
                    .LOCKED_OUT(dcmi_lckd)
	            );

IBUFGDS i_dclkq_bufgds (
	                .I(dclkqP), 
                        .IB(dclkqN), 
                        .O(dclk_qb)
	                );
defparam i_dclkq_bufgds.IOSTANDARD = "LVDS_25";
defparam i_dclkq_bufgds.DIFF_TERM  = "TRUE";


   dcm_dclk dcm_dclkq (
	               .CLKIN_IN(dclk_qb), 
                       .RST_IN(adcdcm_rst), 
		       .CLKDV_OUT(hdclk_q),
                       .CLK0_OUT(dclk_q),
                       .LOCKED_OUT(dcmq_lckd)
	               );

   dcm100 i_dcm100 (.CLKIN_IN(clk100_p), 
                    .RST_IN(fpga_reset), 
                    .CLKDV_OUT(clk6_25), 
                    .CLKIN_IBUFG_OUT(), 
                    .CLK0_OUT(clk100), 
                    .LOCKED_OUT(dcm100_lckd)
		    );

   BUFGMUX i_bufgmux (
		      .I0(hdclk_i),
		      .I1(hdclk_q),
		      .S(adc_opt3[2]),
		      .O(hdclk_actv_mx)
		      );

   dcm_clkmeas i_dcm_cm (.CLKIN_IN(hdclk_actv_mx), 
			 .RST_IN(adcdcm_rst), 
			 .CLK0_OUT(hdclk_actv), 
			 .LOCKED_OUT(hdclk_actv_lckd)
			 );

   adc1k_if adc_if0 (
	             .dclk(dclk_i), 
                     .Dpin_N(diN), 
                     .Dpin_P(diP), 
                     .reset(fpga_reset), 
                     .SET_IN(1'b0),
                     .Dout_1(di_r), 
                     .Dout_2(di_f)
	             );

   adc1k_if adc_if1 (
	             .dclk(dclk_i), 
                     .Dpin_N(didN), 
                     .Dpin_P(didP), 
                     .reset(fpga_reset), 
                     .SET_IN(1'b0), 
		     .Dout_1(did_r), 
                     .Dout_2(did_f)
	             );

   adc1k_if adc_if2 (
	             .dclk(dclk_i), 
                     .Dpin_N(dqN), 
                     .Dpin_P(dqP), 
                     .reset(fpga_reset), 
                     .SET_IN(1'b0), 
                     .Dout_1(dq_r), 
                     .Dout_2(dq_f)
	             );

   adc1k_if adc_if3 (
	             .dclk(dclk_i), 
                     .Dpin_N(dqdN), 
                     .Dpin_P(dqdP), 
                     .reset(fpga_reset), 
                     .SET_IN(1'b0),
                     .Dout_1(dqd_r), 
                     .Dout_2(dqd_f)
	             );

   assign fifo_i_in = {di_sync[47:36],
		       did_sync[47:36],
		       di_sync[35:24],
		       did_sync[35:24],
		       di_sync[23:12],
		       did_sync[23:12],
		       di_sync[11:0],
		       did_sync[11:0]};

   fifo_96_4k fifoI (
	             .din(fifo_i_in),
	             .rd_clk(clk100),
	             .rd_en(fifo_ren_i),
	             .rst(fifo_reset),
	             .wr_clk(hdclk_i),
	             .wr_en(fifo_wr_en),
	             .dout(fifo_i),
	             .empty(fifo_i_mt),
	             .full(fifo_i_full)
		     );

   assign fifo_q_in = {dq_sync[47:36],
		       dqd_sync[47:36],
		       dq_sync[35:24],
		       dqd_sync[35:24],
		       dq_sync[23:12],
		       dqd_sync[23:12],
		       dq_sync[11:0],
		       dqd_sync[11:0]};

   fifo_96_4k fifoQ (
	             .din(fifo_q_in),
	             .rd_clk(clk100),
	             .rd_en(fifo_ren_q),
	             .rst(fifo_reset),
	             .wr_clk(hdclk_i),
	             .wr_en(fifo_wr_en),
	             .dout(fifo_q),
	             .empty(fifo_q_mt),
	             .full(fifo_q_full)
		     );

IBUFDS ORI_IBUFDS_INST  (.I(oriP), 
                         .IB(oriN), 
                         .O(ori)
				        );
defparam ORI_IBUFDS_INST.IOSTANDARD = "LVDS_25";
defparam ORI_IBUFDS_INST.DIFF_TERM = "TRUE";

IBUFDS ORQ_IBUFDS_INST (.I(orqP), 
                        .IB(orqN), 
                        .O(orq));
defparam ORQ_IBUFDS_INST.IOSTANDARD = "LVDS_25";
defparam ORQ_IBUFDS_INST.DIFF_TERM = "TRUE";

IBUFGDS i_extrg_bufgds (
	                    .I(ext_trigP), 
                        .IB(ext_trigN), 
                        .O(hw_trig)
	                   );
defparam i_extrg_bufgds.IOSTANDARD = "LVDS_25";
defparam i_extrg_bufgds.DIFF_TERM  = "TRUE";

//------------------------------ Reg Instantiations -----------------------------//
//
//rgstr #(.N(x)) tst_rg    (rst, clk,    wen,                           d,       q     );
rgstr   #(.N(6)) i_gl_csr  (rst, clk100, (wp & (cy_adr[6:0] == 7'h01)), cy_dt[7:2], gl_csr[7:2]);
rgstr   #(.N(1)) i_gl_srst (rst, clk100, (wp & (cy_adr[6:0] == 7'h01)), cy_dt[0], srst);
rgstr   #(.N(8)) i_tst_rg  (rst, clk100, (wp & (cy_adr[6:0] == 7'h16)), cy_dt, tst_rg);
rgstr   #(.N(4)) i_spimctl (rst, clk100, (wp & (cy_adr[6:0] == 7'h21)), cy_dt[4:1], spi_mctl);
rgstr   #(.N(8)) i_spimcfg (rst, clk100, (wp & (cy_adr[6:0] == 7'h22)), cy_dt, spi_cfg);
rgstr   #(.N(7)) i_dt_csr  (rst, clk100, (wp & (cy_adr[6:0] == 7'h30)), cy_dt[7:1], dt_csr);
rgstr   #(.N(8)) i_dt_cfg  (rst, clk100, (wp & (cy_adr[6:0] == 7'h31)), cy_dt, dt_cfg);
rgstr   #(.N(8)) i_dt_pcnt0(rst, clk100, (wp & (cy_adr[6:0] == 7'h33)), cy_dt, dt_pcnt0);
rgstr   #(.N(8)) i_dt_pcnt1(rst, clk100, (wp & (cy_adr[6:0] == 7'h34)), cy_dt, dt_pcnt1);
rgstr   #(.N(8)) i_dt_addr0(rst, clk100, (wp & (cy_adr[6:0] == 7'h35)), cy_dt, dt_addr0);
rgstr   #(.N(8)) i_dt_addr1(rst, clk100, (wp & (cy_adr[6:0] == 7'h36)), cy_dt, dt_addr1);
rgstr   #(.N(8)) i_dt_addr2(rst, clk100, (wp & (cy_adr[6:0] == 7'h37)), cy_dt, dt_addr2);
rgstr   #(.N(7)) i_adcopt1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h41)), cy_dt[7:1], adc_opt1);
rgstr   #(.N(3)) i_adcopt21(rst, clk100, (wp & (cy_adr[6:0] == 7'h42)), cy_dt[2:0], adc_opt21);
rgstr   #(.N(3)) i_adcopt22(rst, clk100, (wp & (cy_adr[6:0] == 7'h42)), cy_dt[6:4], adc_opt22);
rgstr   #(.N(5)) i_adcopt3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h43)), cy_dt[4:0], adc_opt3[4:0]);
/*
  rgstr #(.N(8)) i_pll0_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h50)),  cy_dt, pll_reg0[7:0]);
  rgstr #(.N(8)) i_pll0_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h51)),  cy_dt, pll_reg0[15:8]);
  rgstr #(.N(8)) i_pll0_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h52)),  cy_dt, pll_reg0[23:16]);
  rgstr #(.N(8)) i_pll0_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h53)),  cy_dt, pll_reg0[31:24]);
  
  rgstr #(.N(8)) i_pll1_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h54)),  cy_dt, pll_reg1[7:0]);
  rgstr #(.N(8)) i_pll1_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h55)),  cy_dt, pll_reg1[15:8]);
  rgstr #(.N(8)) i_pll1_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h56)),  cy_dt, pll_reg1[23:16]);
  rgstr #(.N(8)) i_pll1_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h57)),  cy_dt, pll_reg1[31:24]);

  rgstr #(.N(8)) i_pll2_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h58)),  cy_dt, pll_reg2[7:0]);
  rgstr #(.N(8)) i_pll2_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h59)),  cy_dt, pll_reg2[15:8]);
  rgstr #(.N(8)) i_pll2_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5a)),  cy_dt, pll_reg2[23:16]);
  rgstr #(.N(8)) i_pll2_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5b)),  cy_dt, pll_reg2[31:24]);

  rgstr #(.N(8)) i_pll3_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5c)),  cy_dt, pll_reg3[7:0]);
  rgstr #(.N(8)) i_pll3_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5d)),  cy_dt, pll_reg3[15:8]);
  rgstr #(.N(8)) i_pll3_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5e)),  cy_dt, pll_reg3[23:16]);
  rgstr #(.N(8)) i_pll3_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h5f)),  cy_dt, pll_reg3[31:24]);

  rgstr #(.N(8)) i_pll4_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h60)),  cy_dt, pll_reg4[7:0]);
  rgstr #(.N(8)) i_pll4_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h61)),  cy_dt, pll_reg4[15:8]);
  rgstr #(.N(8)) i_pll4_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h62)),  cy_dt, pll_reg4[23:16]);
  rgstr #(.N(8)) i_pll4_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h63)),  cy_dt, pll_reg4[31:24]);

  rgstr #(.N(8)) i_pll5_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h64)),  cy_dt, pll_reg5[7:0]);
  rgstr #(.N(8)) i_pll5_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h65)),  cy_dt, pll_reg5[15:8]);
  rgstr #(.N(8)) i_pll5_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h66)),  cy_dt, pll_reg5[23:16]);
  rgstr #(.N(8)) i_pll5_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h67)),  cy_dt, pll_reg5[31:24]);

  rgstr #(.N(8)) i_pll6_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h68)),  cy_dt, pll_reg6[7:0]);
  rgstr #(.N(8)) i_pll6_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h69)),  cy_dt, pll_reg6[15:8]);
  rgstr #(.N(8)) i_pll6_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6a)),  cy_dt, pll_reg6[23:16]);
  rgstr #(.N(8)) i_pll6_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6b)),  cy_dt, pll_reg6[31:24]);

  rgstr #(.N(8)) i_pll7_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6c)),  cy_dt, pll_reg7[7:0]);
  rgstr #(.N(8)) i_pll7_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6d)),  cy_dt, pll_reg7[15:8]);
  rgstr #(.N(8)) i_pll7_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6e)),  cy_dt, pll_reg7[23:16]);
  rgstr #(.N(8)) i_pll7_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h6f)),  cy_dt, pll_reg7[31:24]);

  rgstr #(.N(8)) i_pll8_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h70)),  cy_dt, pll_reg8[7:0]);
  rgstr #(.N(8)) i_pll8_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h71)),  cy_dt, pll_reg8[15:8]);
  rgstr #(.N(8)) i_pll8_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h72)),  cy_dt, pll_reg8[23:16]);
  rgstr #(.N(8)) i_pll8_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h73)),  cy_dt, pll_reg8[31:24]);

  rgstr #(.N(8)) i_pll9_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h74)),  cy_dt, pll_reg9[7:0]);
  rgstr #(.N(8)) i_pll9_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h75)),  cy_dt, pll_reg9[15:8]);
  rgstr #(.N(8)) i_pll9_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h76)),  cy_dt, pll_reg9[23:16]);
  rgstr #(.N(8)) i_pll9_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h77)),  cy_dt, pll_reg9[31:24]);

  rgstr #(.N(8)) i_pll10_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h78)),  cy_dt, pll_reg10[7:0]);
  rgstr #(.N(8)) i_pll10_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h79)),  cy_dt, pll_reg10[15:8]);
  rgstr #(.N(8)) i_pll10_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7a)),  cy_dt, pll_reg10[23:16]);
  rgstr #(.N(8)) i_pll10_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7b)),  cy_dt, pll_reg10[31:24]);

  rgstr #(.N(8)) i_pll11_0 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7c)),  cy_dt, pll_reg11[7:0]);
  rgstr #(.N(8)) i_pll11_1 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7d)),  cy_dt, pll_reg11[15:8]);
  rgstr #(.N(8)) i_pll11_2 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7e)),  cy_dt, pll_reg11[23:16]);
  rgstr #(.N(8)) i_pll11_3 (rst, clk100, (wp & (cy_adr[6:0] == 7'h7f)),  cy_dt, pll_reg11[31:24]);
*/
wire [3:0] op_code = dt_csr[7:4];

//----------------------------------Real Work---------------------------------------//

//----------------------------------------------------Data capture
//
   // registering the first level of data out of the IDDR primitives
   always @(posedge fpga_reset or posedge dclk_i)
     if (fpga_reset)
       begin
	  di_reg    <= 24'h0;
	  did_reg   <= 24'h0;
	  
	  dq_reg    <= 24'h0;
	  dqd_reg   <= 24'h0;
       end 
     else
       begin
	  di_reg    <=  {di_f,di_r};
	  did_reg   <=  {did_f,did_r};
	  
	  dq_reg    <=  {dq_f,dq_r};
	  dqd_reg   <=  {dqd_f,dqd_r};
       end

   // time demuxing of the data bus into sample n and sample n+1
   // (di "n" and "n+1" and did "n" and "n+1" for example)
   always @(posedge fpga_reset or posedge dclk_i)
     if (fpga_reset)
       begin
	  demux_wen <= 1'b0;
	  
	  di1_reg   <= 48'h0;
	  di2_reg   <= 48'h0;

	  did1_reg  <= 48'h0;
	  did2_reg  <= 48'h0;
	  
	  dq1_reg   <= 48'h0;
	  dq2_reg   <= 48'h0;

	  dqd1_reg  <= 48'h0;
	  dqd2_reg  <= 48'h0;
       end
     else 
       begin
	  if(demux_wen)
	    begin
	       demux_wen <= 1'b0;
	       
	       di1_reg   <= {di_f,di_r,di_reg};
	       di2_reg   <= di1_reg;

	       did1_reg  <= {did_f,did_r,did_reg};	
	       did2_reg  <= did1_reg;

	       dq1_reg   <= {dq_f,dq_r,dq_reg};
	       dq2_reg   <= dq1_reg;

	       dqd1_reg  <= {dqd_f,dqd_r,dqd_reg};
	       dqd2_reg  <= dqd1_reg;
	    end
	  else
	    begin
	       demux_wen <= 1'b1;
	       
	       di1_reg   <= di1_reg;
	       di2_reg   <= di2_reg;
	       
	       did1_reg  <= did1_reg;
	       did2_reg  <= did2_reg;

	       dq1_reg   <= dq1_reg;
	       dq2_reg   <= dq2_reg;

	       dqd1_reg  <= dqd1_reg;
	       dqd2_reg  <= dqd2_reg;
	    end		  
       end

/* The time order for non-des mode (above) is:
* dqd1_reg[11:0]   dq1_reg[11:0]   dqd1_reg[23:12]  dq1_reg[23:12]
* dqd1_reg[25:24] dq1_reg[35:24] dqd1_reg[47:36] dq1_reg[47:36]
* same for di.
* dqd2_reg, dq2_reg,did2_reg,di2_reg simply fillow dxx1_reg
* FIFO input and output follow order above 
*
*
* The time order for des mode (above) is: (example for 8 bit adc)
* dqd1_reg[7:0]   did1_reg[7:0]   dq1_reg[7:0]   di1_reg[7:0]
* dqd1_reg[15:8]  did1_reg[15:8]  dq1_reg[15:8]  di1_reg[15:8]
* dqd1_reg[23:16] did1_reg[23:16] dq1_reg[23:16] di1_reg[23:16]
* dqd1_reg[31:24] did1_reg[31:24] dq1_reg[31:24] di1_reg[31:24]
*
*/

   fifo_48x16 di_syncfifo (
			   .rst(rst),
			   .wr_clk(dclk_i),
			   .rd_clk(hdclk_i),
			   .din(di2_reg),
			   .wr_en(1'b1),
			   .rd_en(1'b1),
			   .dout(di_sync),
			   .full(),
			   .empty());

   fifo_48x16 did_syncfifo (
			   .rst(rst),
			   .wr_clk(dclk_i),
			   .rd_clk(hdclk_i),
			   .din(did2_reg),
			   .wr_en(1'b1),
			   .rd_en(1'b1),
			   .dout(did_sync),
			   .full(),
			   .empty());
   fifo_48x16 dq_syncfifo (
			   .rst(rst),
			   .wr_clk(dclk_i),
			   .rd_clk(hdclk_i),
			   .din(dq2_reg),
			   .wr_en(1'b1),
			   .rd_en(1'b1),
			   .dout(dq_sync),
			   .full(),
			   .empty());
   fifo_48x16 dqd_syncfifo (
			   .rst(rst),
			   .wr_clk(dclk_i),
			   .rd_clk(hdclk_i),
			   .din(dqd2_reg),
			   .wr_en(1'b1),
			   .rd_en(1'b1),
			   .dout(dqd_sync),
			   .full(),
			   .empty());
   
//-------------------------------------- srd -- gl_csr register bit -- auto-clears
   always @(posedge rst or posedge clk100)

     if (rst)
       begin
//	  srd         <= 1'b0;
	  csr_strt    <= 1'b0;
	  trg_hlck    <= 1'b0;
	  clk_led_trn <= 1'b0;
	  clk_ledd    <= 1'b0;
	  clr_csr_strt  <= 1'b0;
       end
     else 
       begin
//	  srd      <= ( (wp & (cy_adr[6:0] == 7'h01) & cy_dt[1]) | (srd & ~rr_tc));
	  csr_strt <= ( (wp & (cy_adr[6:0] == 7'h30) & cy_dt[0]) | (csr_strt & ~clr_csr_strt));
	  trg_hlck <= ( (wp & (cy_adr[6:0] == 7'h16) & cy_dt[0]) | (trg_hlck & ~clr_trg_hlck));
	  clk_ledd    <= clk_led_cntr[20];
	  clk_led_trn <= clk_led_cntr[20] ^ clk_ledd;

	  // See if ADC->FIFO capture is finished, or FIFO->PC read has finished
	  // If either is finished, clear the capture start bit.
	  clr_csr_strt <= ( (sm_capt == SM_CAPT_ST1) & fifo_full) |
			  (sm_ffr  == SM_FFR_ST2) ;
       end

//  separated rr_tc
  reg hdclk_actv_lckd1, hdclk_actv_lckd2, hdclk_actv_lckd3;
  always @(posedge clk100) begin
    hdclk_actv_lckd1 <= hdclk_actv_lckd;
    hdclk_actv_lckd2 <= hdclk_actv_lckd1;
    hdclk_actv_lckd3 <= hdclk_actv_lckd2;
  end

  wire hdclk_rst = hdclk_actv_lckd && !hdclk_actv_lckd3;
  
  
  wire cntr_rst = rst ;//| hdclk_rst; // | hdclk_actv_lckd;
  
  always @(posedge cntr_rst or posedge clk100)
    if (cntr_rst) begin
      srd <= 1'b0; end
    else begin
      srd      <= ( (wp & (cy_adr[6:0] == 7'h01) & cy_dt[1]) | (srd & ~rr_tc));
    end
  

      
      

//-------------------------------------- capture state machine

   always @(posedge rst or posedge clk100)
     if(rst)
       begin
 	  fifo_i_full_d1 <= 1'b0;
 	  fifo_i_full_d2 <= 1'b0;
	  
 	  fifo_q_full_d1 <= 1'b0;
 	  fifo_q_full_d2 <= 1'b0;

	  fifo_full    <= 1'b0;
       end
     else
       begin
	  fifo_i_full_d1 <= fifo_i_full;
	  fifo_i_full_d2 <= fifo_i_full_d1;

	  fifo_q_full_d1 <= fifo_q_full;
	  fifo_q_full_d2 <= fifo_q_full_d1;
	  
	  fifo_full <= (fifo_i_full_d2 | fifo_q_full_d2);
       end

   always @(posedge rst or posedge clk100)

     if (rst) 
       begin
	  sm_capt       <= SM_CAPT_IDL;
	  fifo_wen      <= 1'b0;
	  fifo_rst      <= 1'b1;
       end
     else 
       begin
	  case (sm_capt)

	  SM_CAPT_IDL:
	    begin
	       if ((op_code == 4'h0) & csr_strt & hw_trgd)
		 begin 
		    fifo_rst <= 1'b1;
		    sm_capt <= SM_CAPT_IDL1;  //transfer data from adc to fifo
		 end
	       else
		 begin
		    fifo_rst <= 1'b0;         //capture request not detected
		    sm_capt <= SM_CAPT_IDL;
		 end
	    end

	  SM_CAPT_IDL1:
	    begin
	       if (fifo_i_mt & fifo_q_mt)
		 begin
		    fifo_rst <= 1'b0;         // FIFOs empty, release reset
		    sm_capt <= SM_CAPT_IDL2;
		 end
	       else
		 begin
		    fifo_rst <= 1'b1;        // FIFOs not empty
		    sm_capt <= SM_CAPT_IDL1;
		 end
	    end
	    
	  SM_CAPT_IDL2:
	    begin   // fifo_full has latency wrt empty assertion, wait here
	       // * IMPORTANT, the fifo should be generated with the fifo full flag reset value = 0 instead of 1
	       // If this is not done, the latency of the clock crossing logic for the reset and full signals
	       // will break this state machine at por.
	       if (~fifo_full) sm_capt <= SM_CAPT_ST1;
	       else sm_capt <= SM_CAPT_IDL2;
	    end
	    
	  SM_CAPT_ST1:
            begin
	       if (fifo_full)
		 begin                       // FIFO is full, stop writing and go to idle
		    fifo_wen <= 1'b0;
		    sm_capt  <= SM_CAPT_ST2;
		 end
	       else
		 begin                      // FIFO still filling
		    fifo_wen <= 1'b1;
		    sm_capt  <= SM_CAPT_ST1;
		 end	       
	    end
	  
	  SM_CAPT_ST2: sm_capt <= SM_CAPT_IDL;  //wait for start bit to clear

	  default:
            sm_capt <= SM_CAPT_IDL;

	endcase
     end

//---------------------------------------------------

   // Syncronizing the fifo control lines from the clk100 domain to hdclk_i
   always @(posedge rst or posedge hdclk_i)
     begin
	if(rst)
	  begin
	     fifo_rst_d1  <= 1'b0;
	     fifo_reset   <= 1'b0;

	     fifo_wen_d1  <= 1'b0;
	     fifo_wr_en   <= 1'b0;
	  end
	else
	  begin
	     fifo_rst_d1 <= fifo_rst;
	     fifo_reset  <= fifo_rst_d1;

	     fifo_wen_d1 <= fifo_wen;
	     fifo_wr_en  <= fifo_wen_d1;
	  end
     end

//--------------------------------------------------- fifo read state machine
 always @(*)
   begin
      if(iq_capt)
	   case (op_code)
	     4'hd: fifo_mux = fifo_i;
	     4'hf: fifo_mux = fifo_q;
	     default: fifo_mux = fifo_i;
	   endcase
      else if(q_ch)
	     fifo_mux = fifo_q;
      else
	fifo_mux = fifo_i;
   end
	     
   
   always @(posedge rst or posedge clk100)
     if (rst)
       begin
	  dt_data_0     <= 8'h0;
	  dt_data_1     <= 8'h0;
	  dt_data_2     <= 8'h0;
	  dt_data_3     <= 8'h0;
	  fifo_ren_i    <= 1'b0;
	  fifo_ren_q    <= 1'b0;
	  fcyc          <= 3'h0;
	  sm_ffr        <= SM_FFR_IDL;
       end
     else
       begin
	if (sm_capt == SM_CAPT_ST2) fcyc <= 3'h0;

	  case (sm_ffr)
	    SM_FFR_IDL:
              if (((op_code == 4'hd) | ((op_code == 4'hf) & iq_capt)) & csr_strt)
		begin
		  if(des_mode) begin
		    if ( (~(|fcyc) )) begin 
                      fifo_ren_i <= 1'b1;
		      fifo_ren_q <= 1'b1;
		    end  
		  end
		  else if (iq_capt) begin
		    if (op_code == 4'hd && (~fcyc[1] & ~fcyc[0] & ~des_mode)) fifo_ren_i <= 1'b1;
		    if (op_code == 4'hf && (~fcyc[1] & ~fcyc[0] & ~des_mode)) fifo_ren_q <= 1'b1;
		  end
		  else begin
		    if ((~fcyc[1] & ~fcyc[0] & ~des_mode)) begin
		      fifo_ren_i <= 1'b1;
                      fifo_ren_q <= 1'b1;
		    end
		  end

		   sm_ffr   <= SM_FFR_ST1;
		end

	    SM_FFR_ST1:
              begin
                 fifo_ren_i  <= 1'b0;
	 	 fifo_ren_q  <= 1'b0;	    
                 sm_ffr    <= SM_FFR_ST2;
              end

	    SM_FFR_ST2:
              begin
		 case (fcyc)
		   3'h0:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[11:0] : fifo_mux[11:0];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[11:0] : fifo_mux[23:12];
		     end

		   3'h1:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[23:12] : fifo_mux[35:24];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[23:12] : fifo_mux[47:36];
		     end

		   3'h2:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[35:24] : fifo_mux[59:48];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[35:24] : fifo_mux[71:60];
		     end

		   3'h3:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[47:36] : fifo_mux[83:72];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[47:36] : fifo_mux[95:84];
		     end
		   
		   3'h4:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[59:48] : fifo_mux[11:0];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[59:48] : fifo_mux[23:12];
		     end
		   
		   3'h5:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[71:60] : fifo_mux[35:24];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[71:60] : fifo_mux[47:36];
		     end
		   
		   3'h6:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[83:72] : fifo_mux[59:48];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[83:72] : fifo_mux[71:60];
		     end
		   
		   3'h7:
		     begin
			{dt_data_1[3:0], dt_data_0} <= des_mode ? fifo_q[95:84] : fifo_mux[83:72];
			{dt_data_3[3:0], dt_data_2} <= des_mode ? fifo_i[95:84] : fifo_mux[95:84];
		     end
		 endcase
		 
                 sm_ffr    <= SM_FFR_ST3;
              end
	    
	    SM_FFR_ST3:
              begin
		 fcyc      <= fcyc + 1'b1;
                 sm_ffr    <= SM_FFR_IDL;
              end

	    default: sm_ffr <= SM_FFR_IDL;
	    
	  endcase
       end

   
//--------------------------------------
//
//--------------------------------------reference rate counter
assign rr_tc = ~|rr_cntr;// & srd;

  // changed rst to hdclk_actv_lckd
always @(posedge cntr_rst or posedge clk100)

  if (cntr_rst) begin
    rr_cntr    <= 16'd50000;	//or 53125
	srd1       <= 1'b0;
  end else begin
	srd1   <= srd;
    if (~srd1)  rr_cntr <= 16'd50000;
	else if (srd & srd1 & ~rr_tc) rr_cntr <= rr_cntr - 1'b1;
  end

//--------------------------------------sample rate counter
  // changed rst to hdclk_actv_lckd
always @(posedge cntr_rst or posedge hdclk_i)//hdclk_actv)

  if (cntr_rst) begin
    sr_cntr    <= 17'h0;
	srd2       <= 1'b0;
	srd3       <= 1'b0;
	rr_tc1     <= 1'b0;
  end else begin
	srd2    <= srd;
	srd3    <= srd2;
	rr_tc1  <= rr_tc;
  
    if (srd2 & ~srd3)  sr_cntr <= 17'h0;
  	else if (srd2 & srd3 & ~rr_tc1) sr_cntr <= sr_cntr + 1'b1;

  end
    
//-----------------------------------------------------------------
//Cypress's read multiplexer

assign cy_data = (cyaddr_vld & ~cs_ & ~rd_) ? cy_dato : 8'hz;

always @(*)

  begin

    case (cy_adr[6:0])

      7'h00: cy_dato = SPEC_VER;
      7'h01: cy_dato = {gl_csr[7:2],srd,srst};
//    7'h14: cy_dato = 8'ha8;
//    7'h15: cy_dato = 8'h61;
      7'h11: cy_dato = FPGA_REV_NUM;
      7'h12: cy_dato = 8'h0;	//fsr1
      7'h13: cy_dato = 8'h4;	//fsr2
      7'h14: cy_dato = {sr_cntr[8:3],dcmi_lckd,pll_ld};
      7'h15: cy_dato = sr_cntr[16:9];
      7'h16: cy_dato = {dcmi_lckd,pll_ld,tst_rg[5:0]};
      7'h17: cy_dato = 8'h04; //fpga version 0.4
      7'h21: cy_dato = {3'h0,spi_mctl,sdo_adc};
      7'h22: cy_dato = spi_cfg;
      7'h30: cy_dato = {dt_csr[7:1],csr_strt};
      7'h31: cy_dato = dt_cfg;
      7'h35: cy_dato = dt_addr0;
      7'h36: cy_dato = dt_addr1;
      7'h37: cy_dato = dt_addr2;
      7'h38: cy_dato = {1'b0, dt_data_0[7:1]};
      7'h39: cy_dato = 8'b0; //dt_data_1;
      7'h3a: cy_dato = {1'b0, dt_data_2[7:1]};
      7'h3b: cy_dato = 8'b0; //dt_data_3;
      7'h40: cy_dato = 8'h0d;		//dut id
      7'h41: cy_dato = {adc_opt1,calrun};
      7'h42: cy_dato = {pll_ld,adc_opt22,calfail,adc_opt21};
      7'h43: cy_dato = {3'h0,adc_opt3[4:0]};
/*
      7'h50: cy_dato = pll_reg0[7:0];
      7'h51: cy_dato = pll_reg0[15:8];
      7'h52: cy_dato = pll_reg0[23:16];
      7'h53: cy_dato = pll_reg0[31:24];
      7'h54: cy_dato = pll_reg1[7:0];
      7'h55: cy_dato = pll_reg1[15:8];
      7'h56: cy_dato = pll_reg1[23:16];
      7'h57: cy_dato = pll_reg1[31:24];
      7'h58: cy_dato = pll_reg2[7:0];
      7'h59: cy_dato = pll_reg2[15:8];
      7'h5a: cy_dato = pll_reg2[23:16];
      7'h5b: cy_dato = pll_reg2[31:24];
      7'h5c: cy_dato = pll_reg3[7:0];
      7'h5d: cy_dato = pll_reg3[15:8];
      7'h5e: cy_dato = pll_reg3[23:16];
      7'h5f: cy_dato = pll_reg3[31:24];
      7'h60: cy_dato = pll_reg4[7:0];
      7'h61: cy_dato = pll_reg4[15:8];
      7'h62: cy_dato = pll_reg4[23:16];
      7'h63: cy_dato = pll_reg4[31:24];
      7'h64: cy_dato = pll_reg5[7:0];
      7'h65: cy_dato = pll_reg5[15:8];
      7'h66: cy_dato = pll_reg5[23:16];
      7'h67: cy_dato = pll_reg5[31:24];
      7'h68: cy_dato = pll_reg6[7:0];
      7'h69: cy_dato = pll_reg6[15:8];
      7'h6a: cy_dato = pll_reg6[23:16];
      7'h6b: cy_dato = pll_reg6[31:24];
      7'h6c: cy_dato = pll_reg7[7:0];
      7'h6d: cy_dato = pll_reg7[15:8];
      7'h6e: cy_dato = pll_reg7[23:16];
      7'h6f: cy_dato = pll_reg7[31:24];
      7'h70: cy_dato = pll_reg8[7:0];
      7'h71: cy_dato = pll_reg8[15:8];
      7'h72: cy_dato = pll_reg8[23:16];
      7'h73: cy_dato = pll_reg8[31:24];
      7'h74: cy_dato = pll_reg9[7:0];
      7'h75: cy_dato = pll_reg9[15:8];
      7'h76: cy_dato = pll_reg9[23:16];
      7'h77: cy_dato = pll_reg9[31:24];
      7'h78: cy_dato = pll_reg10[7:0];
      7'h79: cy_dato = pll_reg10[15:8];
      7'h7a: cy_dato = pll_reg10[23:16];
      7'h7b: cy_dato = pll_reg10[31:24];
      7'h7c: cy_dato = pll_reg11[7:0];
      7'h7d: cy_dato = pll_reg11[15:8];
      7'h7e: cy_dato = pll_reg11[23:16];
      7'h7f: cy_dato = pll_reg11[31:24];
*/
      
      default: cy_dato = 8'hxx;

	endcase
  end
	     

//-----------------------------------------------------------------
always @(posedge rst or posedge clk100)  //write pulse

  if (rst) begin
    wp    <= 1'b0;
    sm_wp <= SM_WP_IDL;
    end else begin

    case (sm_wp)

      SM_WP_IDL:
        if ( (~(cs_ | wr_)) & cyaddr_vld) begin
          wp    <= 1'b1;
	      sm_wp <= SM_WP_ST1;
	    end

      SM_WP_ST1: 
	    begin
          wp    <= 1'b0;
          if ( (cs_ & wr_)) sm_wp <= SM_WP_IDL;
	    end

      default: sm_wp <= SM_WP_IDL;

    endcase
  end


//----------------------------------------------- dcm reset


always @(posedge fpga_reset or posedge clk100)

		if (fpga_reset) begin
		  adcdcm_rst <= 1'b1;
		  sm_dcmr    <= SM_DCMR_IDL;
		end
	    else begin

	      case (sm_dcmr)

		    SM_DCMR_IDL:
		      begin
			    adcdcm_rst <= 1'b0;
			    if (clk_led_cntr[20] & (~dcmi_lckd | ~dcmq_lckd)) begin
			      adcdcm_rst <= 1'b1;
			      sm_dcmr    <= SM_DCMR_ST1;
			    end
		      end

		    SM_DCMR_ST1:
	          sm_dcmr    <= SM_DCMR_ST1A;

		    SM_DCMR_ST1A:
	          sm_dcmr    <= SM_DCMR_ST1B;

		    SM_DCMR_ST1B:
	          sm_dcmr    <= SM_DCMR_ST1C;

		    SM_DCMR_ST1C:
	          sm_dcmr    <= SM_DCMR_ST2;

		    SM_DCMR_ST2:
		      begin
		        adcdcm_rst <= 1'b0;
			    if (~clk_led_cntr[20]) sm_dcmr <= SM_DCMR_IDL;
		      end

            default:
		      sm_dcmr <= SM_DCMR_IDL;

	      endcase
	    end
//------------------------------------------------------------------------------------

always @(posedge fpga_reset or posedge clk100)

		if (fpga_reset) begin
		  sm_pdq       <= 1'b0;
		  sm_hlck      <= SM_HL_IDL;
		  clr_trg_hlck <= 1'b0;
		end
	    else begin

	      case (sm_hlck)

		    SM_HL_IDL:
		      begin
			    sm_pdq <= 1'b0;
			    if (trg_hlck & clk_led_trn) begin
			      sm_hlck    <= SM_HL_ST1;
			    end
		      end

		    SM_HL_ST1:
		      begin
				sm_pdq   <= 1'b1;
	            if (clk_led_trn) sm_hlck <= SM_HL_ST2;
		      end

		    SM_HL_ST2:
		      begin
				sm_pdq   <= 1'b0;
	            if (clk_led_trn) sm_hlck <= SM_HL_ST3;
		      end

		    SM_HL_ST3:
		      begin
	            if (led_overtemp) sm_hlck <= SM_HL_ST1;
				else begin 
				  clr_trg_hlck <= 1'b1;
				  sm_hlck <= SM_HL_ST4;
				end
		      end

		    SM_HL_ST4:
			  begin
				clr_trg_hlck <= 1'b0;
		        if (~trg_hlck) sm_hlck <= SM_HL_IDL;
			  end
            
			default:
		      sm_hlck <= SM_HL_IDL;

	      endcase
	    end

//------------------------------------------------------------------------------------

  always @(posedge fpga_reset or posedge clk6_25)
	if (fpga_reset) begin 
      clk_led_cntr <= 21'h0;
	end else begin 
		clk_led_cntr <= clk_led_cntr + 1'b1;
	end

//-------------------------------------- Cypress' I/O FF
always @(posedge clk100)

 begin
  cy_adr <= cy_addr;
  cy_dt  <= cy_data;
  cs_    <= cs_n;
  rd_    <= rd_n;
  wr_    <= we_n;
 end

//--------------------------------------- ADC SPI

assign scs_adc  = spi_mctl[3] & (spi_cfg[2:0] == 0);
assign sclk_adc = spi_mctl[2] & (spi_cfg[2:0] == 0);
assign sdi_adc  = spi_mctl[1] & (spi_cfg[2:0] == 0);

//--------------------------------------- USI SPI

//assign sdo_usi  = spi_mctl[4] ? (spi_mctl[1] & spi_cfg[2]) : 1'bZ;
assign sdo_usi  = spi_mctl[1] & spi_cfg[2];
assign sclk_usi = spi_mctl[2] & spi_cfg[2];
assign scs1_usi = ~(~spi_mctl[3] & (spi_cfg[2:0] == 3'h4) );
assign scs2_usi = ~(~spi_mctl[3] & (spi_cfg[2:0] == 3'h5) );

//--------------------------------------- 2531 Microwire

assign pll_ce   = ~adc_opt22[2];
  lmk_spi lmk_spi_2541_inst
   (.rstn       (adc_opt22[0]),
    .restart    (1'b0),
    .pclk       (clk100),
    .pen        (1'b0),
    .pwr        (1'b0),
    .paddr      (5'h0),
    .pwdata     (16'h0),
    .prdata     (),
    .d          (pll_sdata),
    .c          (pll_sclk),
    .enl        (pll_le),
    .lmk_sync   (),
    .lmk_done   (lmk_2541_done)
   );

/*
   lmk_spi_reg lmk_spi_2541_inst
   (.REG0       (pll_reg0),
    .REG1       (pll_reg1),
    .REG2       (pll_reg2),
    .REG3       (pll_reg3),
    .REG4       (pll_reg4),
    .REG5       (pll_reg5),
    .REG6       (pll_reg6),
    .REG7       (pll_reg7),
    .REG8       (pll_reg8),
    .REG9       (pll_reg9),
    .REG10      (pll_reg10),
    .REG11      (pll_reg11),

    .rstn       (adc_opt22[0]),
    .restart    (1'b0),
    .pclk       (clk100),
    .pen        (1'b0),
    .pwr        (1'b0),
    .paddr      (5'h0),
    .pwdata     (16'h0),
    .prdata     (),
    .d          (pll_sdata),
    .c          (pll_sclk),
    .enl        (pll_le),
    .lmk_sync   (),
    .lmk_done   (lmk_2541_done)
   );
*/
    
// assign pll_le   = mw_le_reg;
// assign pll_sdata = rom_dat_reg[23];
// assign pll_sclk = mw_clk_reg;
// assign pll_le    = spi_mctl[3] & (spi_cfg[2:0] == 3'h1);
// assign pll_sclk  = spi_mctl[2] & (spi_cfg[2:0] == 3'h1);
// assign pll_sdata = spi_mctl[1] & (spi_cfg[2:0] == 3'h1);
//--------------------------------------- LEDs

  reg [26:0] ori_cnt;
  reg [26:0] orq_cnt;
  reg [26:0] cal_cnt;
  reg 	     ori_dly, orq_dly;
  reg 	     ori_dly2, orq_dly2;
  reg 	     cal_dly, cal_dly2;
  
  
  assign led_standby     = ~adc_pwr; //tst_rg[0]; //ADC POWER
  assign led_acquire     = ~(|cal_cnt); //hdclk_actv;
  assign led_trigger     = ~hwtrg_en;
  assign led_overrange_i = ~(|ori_cnt); //~(ori_cnt[28]); //~ori;
  assign led_overrange_q = ~(|orq_cnt); //~orq;
  assign led_health1     = ~dcm100_lckd || fpga_reset; //FPGA_OPERATIONAL clk_led_cntr[20] & ~fpga_reset;
  assign led_health2     = ~dcmi_lckd;
  assign led_needcal     = ece;
  assign led_overtemp    = ~rcout2_locked;

  assign pdi =      1'bz;
  assign pdq =      sm_pdq;
  assign des =      1'bz;
  assign fsr =      1'bz;
  assign ddrphase = 1'bz;
  assign caldly   = 1'bz;
  assign tpm      = 1'bz;
  assign ndm      = 1'bz;
  assign wss      = 1'bz;
  assign cal      = 1'bz;
  assign poren    = 1'bz;
  assign cy_fd    = 16'hz;
  assign fifoadr  = 2'hz;
  assign testp    = 4'bz;
  assign testn    = 4'bz;
  assign cy_id    = 4'hz;
  assign ext_clk_sel = adc_opt3[0];
  assign cy_valid = 1'bz;
  assign jp_[1]   = calrun;
  

  //led stretching!!

  always @(posedge dclk_i or posedge rst)
    if(rst)
      ori_dly <= 1'b0;
    else
      ori_dly <= ori;

  always @(posedge dclk_q or posedge rst)
    if(rst)
      orq_dly <= 1'b0;
    else
      orq_dly <= ori;

  always @(posedge dclk_i or posedge rst)
    if(rst)
      cal_dly <= 1'b0;
    else
      cal_dly <= calrun;

  always @(posedge dclk_i)
    ori_dly2 <= ori_dly;

  always @(posedge dclk_q)
    orq_dly2 <= orq_dly;

  always @(posedge dclk_i)
    cal_dly2 <= cal_dly;

  always @(posedge dclk_i or posedge rst)
    if(rst)
      ori_cnt <= 27'h0;
    else if ((ori_dly && !ori_dly2) || (|ori_cnt))
      ori_cnt <= ori_cnt + 1;
    else
      ori_cnt <= 27'h0;

  always @(posedge dclk_q or posedge rst)
    if(rst)
      orq_cnt <= 27'h0;
    else if ((orq_dly && !orq_dly2) || (|orq_cnt))
      orq_cnt <= orq_cnt + 1;
    else
      orq_cnt <= 27'h0;
  
  always @(posedge dclk_i or posedge rst)
    if(rst)
      cal_cnt <= 27'h0;
    else if ((cal_dly && !cal_dly2) || (|cal_cnt))
      cal_cnt <= cal_cnt + 1;
    else
      cal_cnt <= 27'h0;
  
endmodule // adc07d1520


module rgstr (rst,clk,wen,d,q);
  parameter N = 1;
  input  [N-1:0] d;
  output [N-1:0] q;
  input rst,clk,wen;

  reg [N-1:0] q;

  always @(posedge clk or posedge rst)
    if (rst) q <= 0;
    else if (wen) q <= d;

  
endmodule // rgstr
